{
*************************************************************************
*                                                                       *
*                                                                       *
*        R P W D A T A  (Reduce Powder DATA for Diffraction)            *
*                                                                       *
*                2D DRAW Interface Graphic Module                       *
*                                                                       *
*             ( ILL Data Base Manager Source File )                     *
*                                                                       *
*                Version  1.1-C  - - 30-Nov-2009                        *
*                                                                       *
*                                by                                     *
*                                                                       *
*                  Pierre Wolfers, Institut Neel                        *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
*************************************************************************

/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////

}
module RPWDATA_LSQ;

%include 'RPWSRC:rpwdata_env' {, list_on};


const
  max_var =                        256;                 { Maximum of space for parameters to fit in a block }

type
  {lsq_flt   =                   double;                Already done in  the rpwdata_env file }

  flt_vec( len: integer ) = array[1..len] of   lsq_flt; { Define vectors of real, integer and pointer of real ... }
  int_vec( len: integer ) = array[1..len] of   integer;
  ptr_vec( len: integer ) = array[1..len] of  ^lsq_flt;
  ipt_vec( len: integer ) = array[1..len] of  ^integer;


  flv_ptr =                   ^flt_vec;                 { ... and there related pointer types }
  inv_ptr =                   ^int_vec;
  ptv_ptr =                   ^ptr_vec;
  ipt_ptr =                   ^ipt_vec;

  lsb_ptr =                   ^lsb_rec;                 { Define a Least squares Block pointer }

  lsb_rec = record                                      { * Define the Least squares Block definition record }
              lsb_nxt:         lsb_ptr;                 { Link to next block }
              lsb_ide,                                  { Least-Squares block identifier }
              lsb_scn,                                  { Number of singularities }
              lsb_ior,                                  { Parameter identifier origine }
              lsb_dim:         integer;                 { Dimension of block }
              lsb_dmp,                                  { Damping factor of the block }
              lsb_mrq:         lsq_flt;                 { Levenberg-Marquardt factor of the block }
              lsb_bvi,                                  { Related Bi, Sh, Sv, Min and Max Vectors }
              lsb_shf,
              lsb_sav,
              lsb_min,
              lsb_max,
              lsb_mat:         flv_ptr;                 { Related block matrix }
              lsb_var,                                  { Related variable and sigma pointer tables }
              lsb_sig:         ptv_ptr
            end;

  prm_ptr =                   ^prm_rec;                 { Define the parameter to fit descriptor pointer }

  prm_rec = record
              prm_nxt:         prm_ptr;                 { Link to next parameter descriptor }
              prm_min,                                  { Allowed Range for the variable }
              prm_max:         lsq_flt;
              prm_val,                                  { Links to parameter value and sigma }
              prm_sig:        ^lsq_flt
            end;



var
  lsb_frs,                                              { Least squares Block list header }
  lsb_lst:              lsb_ptr := nil;

  prm_frs,                                              { Parameter to fit descriptor list header }
  prm_lst:              prm_ptr := nil;

  fdv_tab:              flv_ptr := nil;                 { Pointer to fonction derivate table (when used) }

  scn,                                                  { Singularity count }
  bid,                                                  { Least-Squares Block identifier count }
  npb:                         integer;                 { Number of parameter in the block to build }

  svok:                        boolean;                 { To flags when parameter save was done }

[global]
var

  ndat,                                                 { Number of data }
  np,                                                   { Total number of parameters }
  nsing:                       integer;                 { Number of detected singularities }

  min_diag:          lsq_flt := 1.0e-6;                 { Minimum for a pivot to avoid a singularity }

  chi2eps,                                              { Increasing tolerance for chi2 in Marquardt mode }
  chi2,                                                 { Chi_squared = Goodness of fit }
  lchi2,                                                { Previous Chi_squared }
  nchi2:                       lsq_flt;                 { Chi_squared*(n-v) = Goodness of fit numerator }





procedure OUT_MATRIX( n: integer; var mat, bv: flt_vec );
var
  ip: integer;

begin
  WRITELN;
  WRITELN( '  *** Matrix(', n:0, ',', n:0, ')  and related  vector :' );
  ip := 1;
  for i := 1 to n do
  begin
    for j := 1 to n do
      if j < i then WRITE( ' ':12 )
               else begin  WRITE( ' ', mat[ip]:11:4 ); ip := ip + 1  end;
    WRITELN( '  *  ', bv[i]:11:4 )
  end;
  WRITELN
end OUT_MATRIX;




procedure MATINV( n: integer; var mat, bvc: flt_vec; var vsg: ptr_vec; var scn: integer );
{ Inversion of a symetric matrix in one dimensional compressed form :

  n    is the dimension of matrix,
  mat  is then matrix to use
  bvc  is the constant vector Bi used to clear it when
       some singularities occure,
  vsg  is the sigma pointer vector to set siglma to -1.0
       when the specified variable  is singular. 
}

var
  i, j, k, l, m, kli, kmi, i1, ii, kdm, imax:  integer;
  suma, term, denom:                           lsq_flt;

begin
  scn  :=     0;
  { Matrix triangularization }
  k := 1;
  m := 1;
  repeat                                        { Loop on m }
    imax := m - 1; l := m;
    repeat                                      { Loop on l }
      suma := 0.0;  kli := l; kmi := m;
      if imax > 0 then                          { Sum over i = 1,m-1 mat[l,i]*mat[m,i] }
        for i := 1 to imax do
        begin
          suma := suma + mat[kli] * mat[kmi];
          j := n - i; kli := kli + j; kmi := kmi + j
        end;
      term := mat[k] - suma;                    { Do term = c(l,m) - sum }
      if l <= m then
      begin
        if term > min_diag then
        begin
          denom   := SQRT( term );              { Do mat(m,m) = sqrt( term ) }
          mat[k]  := denom;
          vsg[m]^ :=  0.0                       { Flag this variable as regular }
        end
        else
        begin
WRITELN( ' *** LSQ$Matinv Find a singularity for variable # ', m:4 );
          scn := scn + 1;
          denom := 1.0; mat[k] := denom;        { Set denom to 1.0 and set 0.0 for
                                                  non diagonal corresponding terms }
          for i := k+1 to k+n-m do  mat[i] := 0.0;
          kmi := m;
          for i := 1 to imax do
          begin  mat[kmi] := 0.0; kmi := kmi + n - i  end;
          bvc[m]  :=  0.0;                      { Clear the constant vector }
          vsg[m]^ := -1.0                       { Flag this variable as singular }
        end
      end
      else mat[k] := term / denom;              { Do mat(l,m) = term / mat(m,m) }
      k := k + 1;
      l := l + 1
    until l > n;                                { End of l loop }
    m := m + 1
  until m > n;                                  { End of m loop }

  { Matrix inversion }
  mat[1] := 1.0 / mat[1];
  kdm := 1;
  for l := 2 to n do                            { Step l of b(l,m) }
  begin
    kdm := kdm + n - l + 2;
    term := 1.0 / mat[kdm];                     { Reciprocal of diagonal term }
    mat[kdm] := term;
    kmi := 0; kli := l; imax := l - 1;
    { Step m of b(l,m) }
    for m := 1 to imax do
    begin
      k := kli; suma := 0.0;                    { Sum terms }
      for i := m to imax do
      begin
        ii := kmi + i;
        suma := suma - mat[kli] * mat[ii];
        kli := kli + n - i
      end;
      mat[k] := suma * term;                    { Mult sum * recip diagonal }
      j := n - m;
      kli := k + j;
      kmi := kmi + j
    end
  end;
  { Premultiply lower triangle by transpose }
  k := 1;
  for m := 1 to n do
  begin
    kli := k;
    for l := m to n do
    begin
      kmi := k; imax := n - l + 1; suma := 0.0;
      for i := 1 to imax do
      begin
        suma := suma + mat[kli] * mat[kmi];
        kli := kli + 1; kmi := kmi + 1
      end ;
      mat[k] := suma ;
      k := k + 1
    end
  end
end MATINV;



procedure SOLVE( n: integer; var mat, bvc, bsh: flt_vec; var vsg: ptr_vec; chi2: lsq_flt := 0.0 );
{ Resolution of system after matinv execution for
  symetric matrix in one dimensionalk compressed form.

  *** RESOLV must be called after excution of MATINV ***

  n             is the dimension of matrix,
  mat           is the matrix to use
  bvc           is the related constant vector Bi,
  bsh           is the shift computed vector,
  dia           is the diagonal vector to keep the
}
var
  i, j, ijd, ij: integer;
  pdi:           lsq_flt;

begin
  for i := 1 to n do
  begin
    pdi :=   0.0;
    ij  :=     i;
    ijd := n - 1;
    for j := 1 to n do
    begin
      pdi := pdi + mat[ij] * bvc[j];
      if j < i then
      begin
        ij := ij + ijd; ijd := ijd - 1
      end
      else
      begin
         { Put the M**-1[i,i] term of inverse matrix for sigma computing }
        if (i = j) and (vsg[i]^ >= 0.0) then            { Set only for regular (not with singularity) variable }
          vsg[i]^ := SQRT( mat[ij] * chi2 );
        ij := ij + 1
      end
    end;
    bsh[i] := pdi
  end
end SOLVE;



procedure FREE_PDESCR;
var
  p, q: prm_ptr;

begin
  if prm_frs <> nil then
  begin                                                 { We free any previous existing least squares block }
    p := prm_frs;
    while p <> nil do
    begin
      q := p;
      p := p^.prm_nxt;
      DISPOSE( q )
    end
  end;
  prm_frs := nil;
  prm_lst := nil
end FREE_PDESCR;


[global]
procedure LSQ$INIT;
var
  p, q: lsb_ptr;

begin
  if lsb_frs <> nil then
  begin                                                 { We free any previous existing least squares block }
    p := lsb_frs;
    while p <> nil do
    begin
      q := p;
      with q^ do
      begin
        DISPOSE( lsb_sig );
        DISPOSE( lsb_var );
        DISPOSE( lsb_mat );
        DISPOSE( lsb_max );
        DISPOSE( lsb_min );
        DISPOSE( lsb_sav );
        DISPOSE( lsb_shf );
        DISPOSE( lsb_bvi );
        p := lsb_nxt
      end;
      DISPOSE( q )
    end
  end;
  if prm_frs <> nil then FREE_PDESCR;                   { We free any previous existing least squares block }
  if fdv_tab <> nil then DISPOSE( fdv_tab );            { Free any previously allocate table of Function derivates }
  fdv_tab := nil;

  chi2    := 1.0e+10;                                   { Set very large last chi2 value to be sure to have best }
  lchi2   :=    chi2;
  chi2eps := 1.0e-04;                                   { Set the epsilon of incresing tolerance }
  svok    :=   false;

  lsb_frs := nil;
  lsb_lst := nil;
  bid     :=   0;                                       { Clear the Block identifier count }
  np      :=   0;                                       { Clear the parameters count }
  npb     :=   0;
  nsing   :=   0
end LSQ$INIT;



[global]
function  LSQ$NEW_VPARM( var prm, sig: lsq_flt; min, max: lsq_flt := flt_max ): integer;
var
  top: integer;
  p:   prm_ptr;

begin
  NEW( p );
  with p^ do
  begin
    prm_nxt := nil;
    if (min = flt_max) and (max = flt_max) then min := - min;
    prm_min := min; prm_max := max;
    prm_val := prm"address;
    prm_sig := sig"address
  end;
  if prm_frs = nil then prm_frs := p
                   else prm_lst^.prm_nxt := p;
  prm_lst := p;
  np  := np  + 1;
  npb := npb + 1;
  LSQ$NEW_VPARM := np
end LSQ$NEW_VPARM;




[global]
function  LSQ$NEW_BLOCK( dmp, mrq: lsq_flt ): integer;
var
  p:  lsb_ptr;
  q:  prm_ptr;
  id: integer;

begin
  if npb > 0 then
  begin
    NEW( p );
    bid := bid + 1;
    with p^ do
    begin
      lsb_nxt :=      nil;
      lsb_ide :=      bid;
      lsb_scn :=        0;
      lsb_ior := np - npb;
      lsb_dim :=      npb;
      lsb_dmp :=      dmp;
      lsb_mrq :=      mrq;
      NEW( lsb_bvi, npb );
      NEW( lsb_shf, npb );
      NEW( lsb_sav, npb );
      NEW( lsb_min, npb );
      NEW( lsb_max, npb );
      NEW( lsb_mat, (npb*(npb + 1)) div 2 );
      NEW( lsb_var, npb );
      NEW( lsb_sig, npb );
      q := prm_frs;
      for ii := 1 to npb do
      with q^ do
      begin
        lsb_var^[ii] := prm_val;
        lsb_sig^[ii] := prm_sig;
        lsb_min^[ii] := prm_min;
        lsb_max^[ii] := prm_max;
        q := prm_nxt
      end
    end;
    if lsb_frs = nil then lsb_frs := p
                     else lsb_lst^.lsb_nxt := p;
    lsb_lst := p;
    FREE_PDESCR;
    npb := 0
  end;
  LSQ$NEW_BLOCK := bid
end LSQ$NEW_BLOCK;



[global]
procedure LSQ$SET_DMP_MRQ( id: integer; dmp, mrq: lsq_flt );
var
  bl:  lsb_ptr;
  iv: integer;

begin
  if id <= 0 then id := 1;
  if (dmp <= 0.0) or (dmp > 1.0) then dmp := 1.0;
  if (mrq <  0.0) or (mrq > 1.0) then mrq := 1.0;
  bl := lsb_frs;
  while bl <> nil do
  with bl^ do
  begin
  exit if id = lsb_ide;
    bl := lsb_nxt
  end;
  if bl <> nil then
  with bl^ do
  begin
    lsb_dmp := dmp;
    lsb_mrq := mrq
  end
end LSQ$SET_DMP_MRQ;



[global]
procedure LSQ$INIT_CYCLE;
var
  bl:         lsb_ptr;
  pm,
  pb:         flv_ptr;
  sz:         integer;

begin
  ndat     :=   0;                                      { Clear the Data count }
  nchi2    := 0.0;                                      { Clear the Chi2 Numerator }
  bl := lsb_frs;
  while bl <> nil do
  with bl^ do
  begin
    pb := lsb_bvi;
    pm := lsb_mat;
    sz := lsb_dim;
    for ii := 1 to sz do                                { Clear the matrix and B vector }
    begin
      pb^[ii] := 0.0;
    end;
    for ii := 1 to (sz*(sz + 1)) div 2 do pm^[ii] := 0.0;
    bl := lsb_nxt
  end
end LSQ$INIT_CYCLE;



[global]
procedure LSQ$SET_ELEM( delta, weight: lsq_flt; in_var drvtab: array[size: integer] of lsq_flt );
var
  id, kk, sz: integer;
  dw, dv, d2: lsq_flt;
  bl:         lsb_ptr;
  pm,
  pb:         flv_ptr;

begin
  dw := delta*weight;                                   { Set dw as the weighted delta (typicaly 1/sigma) }
  bl :=    lsb_frs;                                     { Start from first LSQ Block }
  while bl <> nil do                                    { Loop on LSQ Block }
  with bl^ do
  begin
    pb := lsb_bvi;                                      { Get B vector and M matrix address }
    pm := lsb_mat;
    kk := 0;
    for ii := 1 to lsb_dim do                           { Loop on i for each derivate }
    begin
      dv := drvtab[ii]*weight;                          { Set the weighted derivate w*dF/dQi }
      d2 := dv*weight*lsb_mrq;                          { Set the weighted not diagonal 2th derivate (with mrq factor) }
      pb^[ii] := pb^[ii] + dw*dv;                       { Set the contribution to the B[i] vector element }
      kk := kk + 1;
      pm^[kk] := pm^[kk] + SQR( dv );                   { Set the contribution to the M[i,i] Matrix element }
      for jj := ii + 1 to lsb_dim do                    { Loop on j for each derivate }
      begin
        kk := kk + 1;
        pm^[kk] := pm^[kk] + d2*drvtab[jj]              { Set the contribution to the M[i,j] Matrix element }
      end
    end;
    bl := lsb_nxt                                       { Skip to next LSQ Block }
  end;
  ndat := ndat + 1;                                     { Update the count of data }
  nchi2 := nchi2 + SQR( dw )                            { Update the Chi2 Numerator }
end LSQ$SET_ELEM;



[global]
procedure LSQ$INIT_DERV;
begin
  if fdv_tab = nil then NEW( fdv_tab, np );             { Create the derivate table when it is not existing }
  for ii := 1 to np do fdv_tab^[ii] := 0.0              { Initialize the derivate table }
end LSQ$INIT_DERV;



[global]
procedure LSQ$SET_DERV( id: integer; drv: lsq_flt );
begin
  if fdv_tab = nil then LSQ$INIT_DERV;
  if (id >= 1) and (id <= fdv_tab^.len) then
    fdv_tab^[id] := fdv_tab^[id] + drv
end LSQ$SET_DERV;



[global]
procedure LSQ$SET_CONTR( delta, weight: lsq_flt );
begin
  if fdv_tab <> nil then
    LSQ$SET_ELEM( delta, weight, fdv_tab^ )
end LSQ$SET_CONTR;



[global]
function  LSQ$PROCESS( bend, bchg, bmrq: boolean;
                       function CH_PARM( i: integer; v_o, v_n, v_c, v_s, min, max: lsq_flt ): lsq_flt ): boolean;
var
  bl:                  lsb_ptr;
  pm:                  flv_ptr;
  ii, ij, ir, jj, jr:  integer;
  vo, vn, vc:          lsq_flt;

begin
  nsing :=    0;
  lchi2 := chi2;
  if ndat > np then chi2 := nchi2/(ndat - np)
               else chi2 := nchi2;
  bl := lsb_frs;
  while bl <> nil do
  with bl^ do
  begin
(*
OUT_MATRIX( lsb_dim, lsb_mat^, lsb_bvi^ );
*)
    MATINV( lsb_dim, lsb_mat^, lsb_bvi^, lsb_sig^, lsb_scn );
    SOLVE( lsb_dim, lsb_mat^, lsb_bvi^, lsb_shf^, lsb_sig^, chi2 );
    nsing := nsing + lsb_scn;                           { Set the block singularities number }
(*
    if (lsb_scn > 0) and bend then
      OUT_MATRIX( lsb_dim, lsb_mat^, lsb_shf^ );
*)
    if bend then
    begin { Build the correlation matrix }
      ii := 1;
      ij := 2;
      ir := lsb_dim;
      pm := lsb_mat;
      while ir > 1 do
      begin
        jj := ii + ir;
        jr := ir - 1;
        while jr > 0 do
        begin
          vo := ABS( pm^[ii]*pm^[jj] );
          if vo >= 1.0e-4 then pm^[ij] := pm^[ij]/SQRT( vo )
                          else pm^[ij] := 0.0;
          ij := ij + 1;
          jj := jj + jr;
          jr := jr - 1
        end;
        ij := ij +  1;
        ii := ii + ir;
        ir := ir -  1
      end
    end;
    bl := lsb_nxt
  end;


  if bchg and bmrq and svok then
    if chi2 - lchi2 > chi2eps then bchg := false;       { Do not applied the change when the chi2 is to bad }

  if bchg then
  begin
    bl := lsb_frs;
    while bl <> nil do
    with bl^ do
    begin { The new parameter values is required }
      for ii := 1 to lsb_dim do
      begin
        vo := lsb_var^[ii]^;
        lsb_sav^[ii]  := vo;                            { Save the parameter old values }
        vc := lsb_dmp*lsb_shf^[ii];
        vn := vo + vc;
        lsb_var^[ii]^ :=  CH_PARM( ii+lsb_ior, vo, vn, vc, lsb_sig^[ii]^, lsb_min^[ii], lsb_max^[ii] )
      end;
      bl := lsb_nxt
    end;
    svok := true
  end;
  LSQ$PROCESS := bchg
end LSQ$PROCESS;



[global]
procedure LSQ$RESTORE_PARM;
var
  bl: lsb_ptr;

begin
  bl := lsb_frs;
  while bl <> nil do
  with bl^ do
  begin
    for ii := 1 to lsb_dim do
      lsb_var^[ii]^ := lsb_sav^[ii];
    bl := lsb_nxt
  end;
  chi2 := lchi2
end LSQ$RESTORE_PARM;



[global]
function  LSQ$CORREL( ii, jj: integer ): lsq_flt;
var
  bp, bi, bj: lsb_ptr;
  re:         lsq_flt;
  ij:         integer;

begin
  if ii = jj then re := 1.0
  else
  begin
    { Loop to locate the Least-Squares block of the ii'th parameter }
    re := 0.0;                                          { Assume no correlation until shown otherwise }
    bp := lsb_frs;
    ij :=      ii;
    while bp <> nil do                                  { Loop on each diagonal least-squares block }
    with bp^ do
    begin
    exit if ii <= lsb_dim;
      ii := ii - lsb_dim;
      jj := jj - lsb_dim;
      bp := lsb_nxt
    end;
    if bp <> nil then                                   { When the least squares block of ii is found ... }
    with bp^ do
      if (jj > 0) and (jj <= lsb_dim) then              { When the j'th parameter own to the same least-squares block }
      begin
        if jj < ii then
        begin  ij := ii; ii := jj; jj := ij  end;       { Get the up triangular matrix element indexies }
        ij := lsb_dim*(ii - 1) + jj - (ii*(ii - 1)) div 2;
        re := lsb_mat^[ij]
      end
  end;
  LSQ$CORREL := re
end LSQ$CORREL;



end RPWDATA_LSQ.
