{
*************************************************************************
*                                                                       *
*                                                                       *
*        R P W D A T A  (Reduce Powder DATA for Diffraction)            *
*                                                                       *
*                2D DRAW Interface Graphic Module                       *
*                                                                       *
*             ( ILL Data Base Manager Source File )                     *
*                                                                       *
*                Version  1.1-C  - - 30-Nov-2009                        *
*                                                                       *
*                                by                                     *
*                                                                       *
*                  Pierre Wolfers, Institut Neel                        *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
*************************************************************************

/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////

}
module RPW_INT_SEARCH;

%include 'RPWSRC:rpwdata_int_env.pas' {, list_on};

const

  ref_prmdef = ref_prmty[          4.0,         { rfe_r: Reflection Edge detection rate (of sigma background) }
                                   2.5,         { rft_r: Reflection top detection rate (of sigma background) }
                                   0.5,         { rw0_t: Minimum width of reflection (cte. in 2*theta �) }
                                   0.0,         { rw1_t: Tan factor for : rw = rw0_t + rw1*tan( theta ) }
                                   0.2,         { sp0_r: Profil specific parameter #0 for auto. profile }
                                   0.0,         { sp1_r: Profil specific parameter #1 for auto. profile }
                                     3,         { Reflection Margin to add at the domain when it is possible }
                                     5,         { rmd_n: Minimum number of dots to define a reflection }
                                     5,         { rsp_n: Minimum number of dot between two maximum }
                                 false          { rdb_f: Debug flag }
                        ];



[global]
var
  ref_parm: ref_prmty  :=   ref_prmdef;         { Setup for reflection search }



[global]
procedure REF$DEFAULT_SETUP;
var
  ref_desc: [static] pdsc_tab( 10 ) := [ 0, 'Reflection Parameters',  75, 100, [
                               [ 'Rfe_r : Min. sigma detect fac.',     45,  84, 25,  5, prm_flt, nil, 0.0, 20.0, 0.01 ],
                               [ 'Rft_r : Top. sigma detect fac.',     45,  76, 25,  5, prm_flt, nil, 0.0, 20.0, 0.01 ],
                               [ 'Rw0_t : Min. ref. width cte. part',  45,  68, 25,  5, prm_flt, nil, 0.0, 10.0, 0.01 ],
                               [ 'Rw1_t : Min. ref. Tan coef.',        45,  60, 25,  5, prm_flt, nil, 0.0, 10.0, 0.01 ],
                               [ 'Sp0_r : Profil spc. parm. #0',       45,  52, 25,  5, prm_flt, nil, 0.0,  1.0, 0.01 ],
                               [ 'Sp1_r : Profil spc. parm. #1',       45,  44, 25,  5, prm_flt, nil, 0.0,  1.0, 0.01 ],
                               [ 'Rmr_n : Refl. margin in dots.',      45,  36, 25,  5, prm_int, nil,   2,   20 ],
                               [ 'Rmd_n : Min. number in a refl.',     45,  28, 25,  5, prm_int, nil,   2,  100 ],
                               [ 'Rsp_n : Min. number between.',       45,  20, 25,  5, prm_int, nil,   2,  100 ],
                               [ 'Debug : Debug to help for setting',  65,  12,  5,  5, prm_bool, nil           ]
                           ] ];

  binit:     [static] boolean := true;

begin
  if binit then
  begin
    with ref_desc, ref_parm do
    begin
      SET_REF( tb[ 1].rf, rfe_r );
      SET_REF( tb[ 2].rf, rft_r );
      SET_REF( tb[ 3].rf, rw0_t );
      SET_REF( tb[ 4].rf, rw1_t );
      SET_REF( tb[ 5].rf, sp0_r );
      SET_REF( tb[ 6].rf, sp1_r );
      SET_REF( tb[ 7].rf, rmr_n );
      SET_REF( tb[ 8].rf, rmd_n );
      SET_REF( tb[ 9].rf, rsp_n );
      SET_REF( tb[10].rf, rdb_f )
    end;
    setup_tab[SEA_IDX] := ref_desc"address;
    binit := false
  end;
  ref_parm := ref_prmdef
end REF$DEFAULT_SETUP;




function  CREATE_PEAK( pz: zon_ptr; th, wd: real ): pck_ptr;
var
  pk, p1, p2: pck_ptr;


  procedure PEAK_INSERT( pk: pck_ptr; var pf, pl: pck_ptr; bz: boolean );
  var
    p1, p2: pck_ptr;

  begin
    { Locate the place to insert }
    p1 := nil;
    p2 := pf;
    while p2 <> nil do
    begin
    exit if pk^.thpos < p2^.thpos;
      p1 := P2;
      if bz then p2 := p2^.znxt
            else p2 := p2^.next
    end;
    { Insert the peak }
    if p1 = nil then
    begin { Insert to begin of the list }
      if bz then pk^.znxt := pf
            else pk^.next := pf;
      pf := pk
    end
    else { Not at the begin of list, it can be at the end of list }
      if bz then begin  pk^.znxt := p2; p1^.znxt := pk  end
            else begin  pk^.next := p2; p1^.next := pk  end;

    if bz then begin  if pk^.znxt = nil then pl := pk  end
          else begin  if pk^.next = nil then pl := pk  end
  end PEAK_INSERT;


begin { CREATE_PEAK }
  if peak_free <> nil then
  begin  pk := peak_free; peak_free := pk^.next  end
  else NEW( pk );
  with pk^ do
  begin
    znxt   := nil;
    next   := nil;
    zone   :=  pz;
    thpos  :=  th; sthpo := 0.0; savth :=  th;
    heigh  := 0.0; shigh := 0.0; savhg := 0.0;
    width  :=  wd; swidt := 0.0; savwd :=  wd;
    mixfc  := 0.0; smixf := 0.0;
    intens := 0.0;
    sigma  := 0.0;
    idthe  :=   0;
    idhig  :=   0;
    idwid  :=   0;
    idpr0  :=   0;
    idpr1  :=   0;
    with hkl_parm do
      if (fnc_n >= 0) and
         (fnc_n <= ORD( fnc_type"last ) ) then ftyp := fnc_type( fnc_n )
                                          else ftyp := fnc_gauss;
    enaflg := true
  end;
  { Insert it in the zone packet list }
  with pz^ do
  begin
    npeak := npeak + 1;                         { Increment the zone enable peak count }
    PEAK_INSERT( pk, pckf, pckl, true )
  end;
  PEAK_INSERT( pk, peak_first, peak_last, false );
  { Update the peak count }
  ntotpck := ntotpck + 1;
  peak_curr := pk;
  CREATE_PEAK := pk
end CREATE_PEAK;



[global]
procedure SORT_PEAK;
const
  nch_msg = '*** RPW Warning: RPW-bug: The Reflection Lines count is not correct';

var
  peak_tab:           ^pck_tab;
  pck, pcp, tmp:       pck_ptr;
  np:                  integer;

begin
  str_msg.length := 0;
  if ntotpck > 1 then
  begin
    NEW( peak_tab,  ntotpck );                  { Allocate the peak table }
    { Put all peak pointer in the peak table }
    np := 0;
    pck := peak_first;
    while (pck <> nil) and (np < ntotpck) do
    begin
      np := np + 1; peak_tab^[np] := pck;
      pck := pck^.next
    end;
    { Check the consistence of peak number with the peak count }
    if (pck <> nil) or (np <> ntotpck) then
    begin
      str_msg := nch_msg;
      ntotpck := np
    end;
    { Peak Sort }
    for ii := 1 to ntotpck-1 do
      for jj := ii+1 to ntotpck do
        if peak_tab^[jj]^.thpos < peak_tab^[ii]^.thpos then
        begin
          tmp := peak_tab^[jj];
          peak_tab^[jj] := peak_tab^[ii];
          peak_tab^[ii] := tmp
        end;
    { Restore the peak list }
    pcp := peak_tab^[1];
    peak_first := pcp;
    for ii := 2 to np do
    begin
      pck := peak_tab^[ii];
      pcp^.next := pck;
      pcp := pck
    end;
    pcp^.next := nil;
    peak_last := pcp
(*
;
pck := peak_first;
while pck <> nil do
with pck^ do
begin
  WRITELN( ' Ref Peak at ', thpos:8:2 );
  pck :=next
end
*)
  end
end SORT_PEAK;



procedure FREE_PEAK( p: pck_ptr );
var
  p1, p2: pck_ptr;
  pz:     zon_ptr;
  ph:     hkl_ptr;

begin
  if p <> nil then
  begin
    { Unlink from zone_list }
    with p^.zone^ do
    begin
      ph := hklf;
      while ph <> nil do                        { Loop to clear any HKL peak attachment to our peak }
      begin
        if ph^.pckp = p then ph^.pckp := nil;
        ph := ph^.next
      end;
      p1 := pckf; p2 := nil;
      while (p1 <> nil) and (p1 <> p) do
      begin  p2 := p1; p1 := p1^.znxt  end;
      if (p1 <> nil) and (p1 = p) then
      begin
        if p2 = nil then pckf := p^.znxt
                    else p2^.znxt := p^.znxt;
        if p = pckl then pckl := p2
      end;
      if p^.enaflg then nenapck := nenapck - 1; { Decrement the enable peak count when this peak was enabled }
      npeak := npeak - 1                        { Decrement the zone enable peak count }
    end;

    { Unlink from main peak list }
    p1 := peak_first; p2 := nil;
    while (p1 <> nil) and (p1 <> p) do
    begin  p2 := p1; p1 := p1^.next  end;
    if (p1 <> nil) and (p1 = p) then
    begin
      if p2 = nil then peak_first := p^.next
                  else p2^.next := p^.next;
      if p = peak_last then peak_last := p2
    end;
    { Update the peak count }
    ntotpck := ntotpck - 1;
    { Put in free list }
    p^.next := peak_free;
    peak_free := p
  end
end FREE_PEAK;



procedure FREE_ALL_PEAK( dethkl: boolean := true );
var
  p1, p2:      pck_ptr;
  pz:          zon_ptr;
  ph:          hkl_ptr;

begin
  if (peak_first <> nil) and (peak_last <> nil) then
  begin
    if dethkl then                              { Detach all HKL when required (the default) }
    begin
      ph := hkl_first;
      while (ph <> nil) do begin  ph^.pckp := nil; ph := ph^.next  end
    end;
    pz := zone_first;                           { Loop on all Zone to erase any peak reference }
    while pz <> nil do
    with pz^ do
    begin
      pckf := nil; pckl := nil; npeak := 0;
      pz := next
    end;
    peak_last^.next := peak_free;               { Put all peak record in the free peak list }
    peak_free  := peak_first;
    peak_last  := nil;
    peak_first := nil
  end;
  nenapck := 0;                                 { Clear all peak counts }
  ntotpck := 0
end FREE_ALL_PEAK;



function  CREATE_ZONE( lfl, ril: integer; pz: zon_ptr := nil ): zon_ptr;
{ Create a new integration zone and if pz is specified,
  insert the new zone just before the pz pointed zone,
  else append the new zone at the end of zone list. }
var
  p, q, r: zon_ptr;

begin
  if zone_free <> nil then
  begin  p := zone_free; zone_free := p^.next  end
  else NEW( p );
  with ref_parm, p^ do
  begin
    next   := nil;
    pckf   := nil; pckl   := nil;
    hklf   := nil; hkll   := nil;
    npeak  :=   0;
    lfmar  :=   1; rimar  :=   1;
    lflim  := lfl; rilim  := ril;
    izpr0  :=   0; izpr1  :=   0;
    izwi0  :=   0; izwi1  :=   0;
    thmin  := sel_pat^.dat[lfl].theta;
    thmax  := sel_pat^.dat[ril].theta;
    bchi2  :=-1.0; rchi2  :=-1.0;
    zwidth := 0.0;
    zbkv   :=   0.0; zbks   :=   0.0;
    ztint  :=   0.0; ztsint :=   0.0;
    zprf0  := sp0_r; zprf1  := sp1_r;
    zspr0  :=   0.0; zspr1  :=   0.0;
    zwid0  := 0.5*rw0_t; zwid1  := 0.5*rw1_t;
    zswi0  :=   0.0; zswi1  :=   0.0;
    for ii := 1 to max_bckparm do
    begin  bcf[ii] := 0.0; bcs[ii] := -1.0; bsav[ii] := 0.0  end
  end;
  if p^.next <> nil then WRITELN( '5 true' );
  nzone := nzone + 1;
  if pz = nil then
  begin
    if zone_last = nil then zone_first := p
                       else zone_last^.next := p;
    zone_last := p
  end
  else
  begin
    q := zone_first;
    r := nil;
    while (q <> nil) and (q <> pz) do           { Find the Zone previous to the new one }
    begin  r := q; q := q^.next  end;
    if q = pz then
    begin                                       { When it is found, ... }
      p^.next := q;                             { ... link the new zone to the pz^ zone, ... }
      if r = nil then zone_first := p           { ... and as the new head when pz was the first one, ... }
                 else r^.next := p              { ... else link the new zone to the previous of pz^ one.}
    end
  end;
  CREATE_ZONE := p
end CREATE_ZONE;



procedure FREE_ZONE( p: zon_ptr );
var
  p1, p2: zon_ptr;
  k1, k2: pck_ptr;
  ph:     hkl_ptr;

begin
  if p <> nil then
  begin
    p1 := zone_first; p2 := nil;                { Locate the Zone in the zone list }
    while (p1 <> nil) and (p1 <> p) do
    begin  p2 := p1; p1 := p1^.next  end;

    if (p1 <> nil) and (p1 = p) then            { When the zone is found }
    begin
      k1 := p^.pckf;                            { Loop to free all peaks of the zone }
      while k1 <> nil do
      begin
        k2 := k1;
        k1 := k1^.znxt;
        FREE_PEAK( k2 )
      end;
      ph := p^.hklf;                            { Suppress any HKL attachment to the zone }
      while ph <> nil do
      begin
        ph^.pckp := nil; ph^.zonp := nil;
        ph := ph^.next
      end;
      p^.hklf := nil; p^.hkll := nil;
      nzone := nzone - 1;                       { Update the total zone number }
      if p2 = nil then zone_first := p^.next
                  else p2^.next := p^.next;
      p^.next := zone_free;
      zone_free := p
    end
  end
end FREE_ZONE;



procedure FREE_ALL_ZONE;
var
  ph:  hkl_ptr;

begin
  ph := hkl_first;                                  { Detach all HKl to ALL zone and PEaks }
  while (ph <> nil) do
  begin                                             { Suppress any HKL attachment to the zone and peak of zone }
    ph^.pckp := nil; ph^.zonp := nil;
    ph := ph^.next
  end;
  if peak_first <> nil then FREE_ALL_PEAK( false );
  zone_last^.next := zone_free;
  zone_free  := zone_first;
  zone_last  := nil;
  zone_first := nil;
  nzone := 0
end FREE_ALL_ZONE;



procedure ADJUST_PEAK_WIDTH( lth, rth: real; pkh: pck_ptr );
var
  td:        real;
  p1, p2: pck_ptr;

begin
  p1 := nil;
  p2 := pkh;
  while p2 <> nil do
  with p2^ do
  begin
    if p1 = nil then
    begin
      td := (thpos - lth)/2.5;
      if width > td then width := td
    end
    else
    begin
      td := (thpos - p1^.thpos)/3.0;
      if td < p1^.width then p1^.width := td;
      if td < width then width := td
    end;
    p1 :=   p2;
    p2 := znxt
  end;
  if p1 <> nil then
  with p1^ do
  begin
    td := (rth - thpos)/2.5;
    if td < width then width := td
  end
end ADJUST_PEAK_WIDTH;



function  SET_REFLECTION_MARK( il, ir: integer ): zon_ptr;
{ Create the reflection list }
var
  dlg, ii, im, ip, jj, lip, mip, npk, rip:      integer;
  lmax, lmin, mmax, prv:                           real;
  pz:                                           zon_ptr;

  procedure MAX_SEARCH( il, ir: integer );
  var
    ii, ij, jj, im, ip, nb:   integer;
    lmax, lmin, cv, sv, ww, th:  real;

  begin
    with ref_parm, sel_pat^ do
    if ir - il >= rmd_n then
    begin
      { Look at the maximum in the interval }
      lmax := 0.0;
      for it := il + dlg to ir - dlg do
        with dat[it] do
        if not (mk_invalid in mflg) then                { We exclude any invalid dots }
          if lmax < int then
          begin  lmax := int; ip := it  end;

      if ip > dim then ip := dim                        { Check for pattern limits }
                  else if ip < 0 then ip := 1;

      with dat[ip] do
      begin
        th := theta;
        ww := rw0_t + rw1_t*TAN( th2brr*th )            { Get the reflection Half-width }
      end;
      lip := ip - 1; rip := ip + 1; nb := 1; sv := ip;
      ii := ip - 1;
      while (ii >= il) do
      with dat[ii] do
      begin
        if int  + rft_r*sig >= lmax then
        begin
          sv := sv + ii; nb := nb + 1; lip := ii        { Compute the barycentre in dot }
        end;
      exit if theta < th - ww;
        ii := ii - 1
      end;
      ii := ip + 1;
      while (ii <= ir) do
      with dat[ii] do
      begin
        if int  + rft_r*sig >= lmax then
        begin
          sv := sv + ii; nb := nb + 1; rip := ii        { Compute the barycentre in dot }
        end;
      exit if theta > th + ww;
        ii := ii + 1
      end;
(*
WRITELN( ' ***  il  lip   ip  rip   ir ',  il:5, lip:5, ip:5, rip:5, ir:5 );
*)
      if nb > 0 then
      begin
        ip := ROUND( sv/nb );
        if (rip - il > dlg) and (ir - lip > dlg) then   { We have found a true maximum }
        begin
          with dat[ip] do
          begin
            mflg := mflg+[mk_rflcen];                   { Put a Reflexion Position mark at this maximum }
            if rdb_f then
              WRITELN( ' il, lip, ip, rip, ir, th ', il:5, lip:5, ip:5, rip:5, ir:6, theta:8:3 );
            CREATE_PEAK( pz, theta, rw0_t + rw1_t*TAN( th2brr*theta ) )
          end;
          npk := npk + 1
        end;

        { Form then left search domain }
        if dat[lip].theta - dat[il].theta > ww then
(*      if lip - il > rsp_n then *)
        begin
          cv := lmax; lmin := lmax;
          im := 0; ii := ip - dlg - 1; ij := 0;
          while ii > il do
            with dat[ii] do
            begin
              if not (mk_invalid in mflg) then          { We exclude any invalid dots }
              begin
                if int < lmin then 
                begin
                  lmin := int; cv := lmin; im   :=  ii; ij := 0
                end
                else
                if int > cv then
                begin                                   { Keep the mark of the scan }
                  cv := int; ij := ii;
          exit if (ij > 0) and (im - ij >= dlg);
                end
              end;
              ii := ii - 1
            end;
          if (im > 0) and (im - ij >= dlg) and (ij > 0) then
          begin
(*
WRITELN( ' im, il, ij ', im:5, il:5, ij:5, ' before ', dat[ip].theta:8:3 );
*)
            MAX_SEARCH( il, im )                        { When a enough large space is found before the minimum ... }
                                                        { ... We perform a new search of reflection }
          end
        end;

        { Form then right search domain }
        if dat[ir].theta - dat[rip].theta > ww then
(*      if ir - rip > rsp_n then *)
        begin
          cv := lmax; lmin := lmax;
          im := 0; ii := ip + dlg + 1; ij := 0;
          while ii < ir do
            with dat[ii] do
            begin
              if not (mk_invalid in mflg) then          { We exclude any invalid dots }
              begin
                if int < lmin then 
                begin
                  lmin := int; cv := lmin; im  := ii; ij := 0
                end
                else
                if int > cv then
                begin                                   { Keep the mark of the scan }
                  cv := int; ij := ii;
          exit if (ij > 0) and (ij - im >= dlg);
                 end
              end;
              ii := ii + 1
            end;
          if (im > 0) and (ij - im >= dlg) and (ij > 0) then 
          begin
(*
WRITELN( ' im, ir, ij ', im:5, ir:5, ij:5, ' after  ', dat[ip].theta:8:3 );
*)
            MAX_SEARCH( im, ir )                        { When a enough large space is found before the minimum ... }
                                                        { ... We perform a new search of reflection }
          end
        end
      end
    end
  end MAX_SEARCH;



begin { SET_REFLECTION_MARK }
  if sel_pat <> nil then
  with ref_parm, sel_pat^ do
  begin
    { Look for the main maximum }
    mmax := 0.0;
    ip   :=  il;
    dlg  := rsp_n div 2;
    if dlg < 2 then dlg := 2;
    { Set up domain limit marks }
    with dat[il] do mflg := mflg + [mk_rfleft];
    with dat[ir] do mflg := mflg + [mk_rfright];
    pz := CREATE_ZONE( il, ir );
    if ref_parm.rdb_f then
      with pz^ do
        WRITELN( ' Create a the Zone # ', nzone:4, ' from ', thmin:8:2, ' to ', thmax:8:2,
                 ' with indices in range [', il:0, '..', ir:0, ']' );
    npk := 0;
    MAX_SEARCH( il, ir );

    { Now we adjust the width of reflection with distance between peaks and wzone limits }
    if pz <> nil then
    with pz^ do
      ADJUST_PEAK_WIDTH( dat[il].theta, dat[ir].theta, pckf )
  end;
  SET_REFLECTION_MARK := pz
end SET_REFLECTION_MARK;



procedure LOCATE_REFLECTION( ifrs, ilst: integer );
var
  ib, ii, il, ir, ip, siz: integer;
  ca, cb, yb:                 real;
  br, bok:                 boolean;
  pz:                      zon_ptr;

  xx, yy:       array[1..2] of real;

begin
  if zone_first <> nil then FREE_ALL_ZONE;              { We must destroye any previous reflection packet records }
  nzone  :=     0;
  bok    := false;
  if (sel_pat <> nil) and (bckgrd_crv <> nil) then
  with ref_parm, sel_pat^, bckgrd_crv^ do
  begin
    if rmd_n < rsp_n then rsp_n :=  rmd_n;
    br  :=    false;                                    { We begin out of a reflection }
    siz := 2*crv_sz;
    xx[1] := crv_tab[1]; yy[1] := crv_tab[2] - cshift;
    xx[2] := crv_tab[3]; yy[2] := crv_tab[4] - cshift; ib := 4;
    ca := (yy[2] - yy[1])/(xx[2] - xx[1]);
    cb := yy[2] - ca*xx[2];
    for ip := ifrs to ilst do
    with dat[ip] do
      if not (mk_invalid in mflg) then                  { We exclude any invalid dots }
      begin
        mflg := mflg - [mk_rfleft,mk_rflcen,mk_rflpos,mk_rflhkl,mk_rfright];
        if (theta > xx[2]) and (ib < siz) then
        begin
          xx[1] := xx[2]; yy[1] := yy[2];               { We get a new background equation }
          ib := ib + 1; xx[2] := crv_tab[ib];
          ib := ib + 1; yy[2] := crv_tab[ib] - cshift;
          ca := (yy[2] - yy[1])/(xx[2] - xx[1]);
          cb := yy[2] - ca*xx[2]
        end;
        yb := ca*theta + cb;                            { Compute the entered/generated Background }
        if ((int - yb) > rfe_r*sig) then                { The dot is significatively higher than the background }
        begin { When we are not in a reflection scan, It is possible that is a new one }
          ir := 0;                                      { Clear the right index for minimum packet spacing }
          if bok and not br then
          begin  il := ip - 1; br := true  end          { Keep Memory of begin pattern index and set the flag }
        end
        else
        begin
          if br then                                    { When we are can finish a reflection scan }
          begin
            if ip - il >= rmd_n then                    { The minimum of reflection dot is reached }
            begin
              if ir = 0 then ir := ip;                  { Keep the end of packet index }
              if ip - ir >= 2 then                      { The minimum spacing (= margin) is present }
              begin
                pz := SET_REFLECTION_MARK( il, ir );
{              if pz <> nil then npeaks := npeaks + pz^.npeak; }
                br := false
              end
            end else br := false                        { Minimum not reached => it is not a reflection line }
          end;
          bok := true
        end
      end;

    { Now we can add margins when it is possible }
    if {(npeaks > 0) and} (zone_first <> nil) then
    begin
      ip := ifrs;
      ib := ifrs;
      pz := zone_first;
      repeat
        { Look for a left mark }
        while (ip < ilst) and (not (mk_rfleft in dat[ip].mflg)) do  ip := ip + 1;
      exit if ip >= ilst;
        { Left mark is found }
        if not (mk_rfright in dat[ip].mflg) then
        begin { For single mark only }
          il := ip - rmr_n;
          if il < ib then il := ib;                     { Set the best limit position }
          while (il < ip) and (mk_invalid in dat[il].mflg) do  il := il + 1;
          if il < ip then
          begin                                         { We displace the left marker }
            with dat[ip] do mflg := mflg - [mk_rfleft];
            with dat[il] do mflg := mflg + [mk_rfleft];
            with pz^ do
            begin  lfmar := ip - il + 1; lflim := il  end
          end
        end;
        ip := ip + 1;
        { We look for the right marker }
        while (ip < ilst) and (not (mk_rfright in dat[ip].mflg)) do  ip := ip + 1;
      exit if ip >= ilst;
        ib := ip;
        if not (mk_rfleft in dat[ip].mflg) then
        begin { For single mark only }
          il := ip + 2*rmr_n; if il > ilst then il := ilst;
          ib := ip + 1;
          while (ib < il) and (not (mk_rfleft in dat[ib].mflg)) do  ib := ib + 1;
          if ib - ip < 2*rmr_n then ib := ip + (ib - ip) div 2
                               else ib := ip + rmr_n;
          while (ib > ip) and (mk_invalid in dat[il].mflg) do ib := ib - 1;
          if ib > ip then
          begin                                         { We displace the right marker }
            with dat[ip] do mflg := mflg - [mk_rfright];
            with dat[ib] do mflg := mflg + [mk_rfright];
            with pz^ do
            begin  rimar := ib - ip + 1; rilim := ib  end
          end;
          ip := ib
        end;
        pz := pz^.next
      until ip >= ilst;
    end
  end;
  nenapck := ntotpck;           { Current number of Enabled peaks }
  nerr_flg := false             { Clear the Fit computing error flag }
end LOCATE_REFLECTION;




[global]
function  INTEGR$REF_LOCATE: ^string;
begin
  bfitini     :=  true;                                 { Set the init mode for the next reflection fit }
  str_msg.length  := 0;
  if (sel_pat <> nil) and (sel_iright > sel_ileft) and (bckgrd_crv <> nil) then
    LOCATE_REFLECTION( sel_ileft, sel_iright );
  if fitcmp_crv <> nil then
  begin
    DEL_CURVE( fitcmp_crv );
    fitcmp_crv := nil
  end;
  if str_msg.length > 0 then INTEGR$REF_LOCATE := str_msg"address
                        else INTEGR$REF_LOCATE := nil
end INTEGR$REF_LOCATE;



[global]
procedure MOVE_PEAK_POS( pk: pck_ptr; fhkl: mrk_flgtyp := [] );
var
  t0, t1:         real;
  it:          integer;

begin
  if fhkl = [] then fhkl := [mk_rflpos,mk_rflcen]
               else fhkl := fhkl*[mk_rflpos,mk_rflcen,mk_rflhkl];
  if pk <> nil then
  with pk^, zone^, sel_pat^ do
  begin
    it := lflim;
    t0 := dat[it].theta; t1 := t0;
    repeat
      it := it + 1;
      t0 := t1; t1 := dat[it].theta
    until (it >= rilim) or (t1 > thpos);
    if ABS( thpos - t0 ) < ABS( thpos - t1 ) then it := it - 1;
    dat[it].mflg := dat[it].mflg + fhkl
  end
end MOVE_PEAK_POS;



[global]
procedure ADJUST_PEAK_POS( pz: zon_ptr; fhkl: mrk_flgtyp := [] );
{ Put the reflection mark of a zone at there fitted positions }
var
  pk:          pck_ptr;
  t0, t1:         real;
  it:          integer;

begin
  if fhkl = [] then fhkl := [mk_rflpos,mk_rflcen]
               else fhkl := fhkl*[mk_rflpos,mk_rflcen,mk_rflhkl];
  with pz^, sel_pat^ do
  begin
    for ii := lflim to rilim do
      dat[ii].mflg := dat[ii].mflg - [mk_rflpos,mk_rflcen,mk_rflhkl];

    pk :=  pckf;
    while pk <> nil do
    with pk^ do
    begin
      it := lflim;
      t0 := dat[it].theta; t1 := t0;
      repeat
         it := it + 1;
         t0 := t1; t1 := dat[it].theta
      until (it >= rilim) or (t1 > thpos);
      if ABS( thpos - t0 ) < ABS( thpos - t1 ) then it := it - 1;
      dat[it].mflg := dat[it].mflg + fhkl;
      pk := znxt
    end
  end
end ADJUST_PEAK_POS;



[global]
function  INTEGR$REF_COPY: ^string;
{ To set the pattern zone and peak of an other pattern }
const
  tol = 0.25;

var
  pz:          zon_ptr;
  pk:          pck_ptr;
  np, ip:      integer;
  berr:        boolean;


  procedure SET_TO_VALID( ip: integer );
  const
    iexpl = 5;

  var
    i1, i2, ie: integer;

  begin
    with sel_pat^, dat[ip] do
    begin
      i1 := ip - 1;
      ie := ip - iexpl; if ie < sel_ileft then ie := sel_ileft;
      while i1 >= ie do
      begin
      exit if not (mk_invalid in dat[i1].mflg);
        i1 := i1 - 1
      end;
      if i1 < ie then i1 := ip;
      i2 := ip + 1;
      ie := ip + iexpl; if ie > sel_iright then ie := sel_iright;
      while i2 <= ie do
      begin
      exit if not (mk_invalid in dat[i2].mflg);
        i2 := i2 + 1
      end;
      if i2 > ie then i2 := ip;
      if i1 <> i2 then
      begin
        mflg := mflg - [mk_invalid];
        if i1 = ip then begin  int := dat[i2].int; sig := dat[i2].sig  end
        else
        if i2 = ip then begin  int := dat[i1].int; sig := dat[i1].sig  end
        else
        begin
          int := 0.5*(dat[i1].int + dat[i2].int);
          sig := 0.5*(dat[i1].sig + dat[i2].sig)
        end
      end
    end
  end SET_TO_VALID;


begin { INTEGR$REF_COPY }
  if (zone_first <> nil) and (sel_pat <> nil) then
  with sel_pat^ do
  begin
    berr   :=    false;
    pz   := zone_first;
    np     :=        0;
    while pz <> nil do
    with pz^ do
    begin
      with dat[lflim] do
      begin
        if mk_invalid in mflg then SET_TO_VALID( lflim );
        mflg := mflg + [mk_rfleft]
      end;
      with dat[rilim] do
      begin
        if mk_invalid in mflg then SET_TO_VALID( rilim );
        mflg := mflg + [mk_rfright]
      end;
      pk := pckf;
      while pk <> nil do
      with pk^ do
      begin
        ip := LOCATE_ANGLE( thpos, tol, lflim, true, true );
        if (ip < 0) or (ip > rilim) then enaflg := false
        else
        with dat[ip] do
        begin
          mflg := mflg + [mk_rflpos];
          np := np + 1; enaflg := true
        end;
        pk := znxt
      end;
      pz := next
    end
  end;
  nenapck := np;
  INTEGR$REF_COPY := nil
end INTEGR$REF_COPY;



procedure RELOAD_ZPCK( pz: zon_ptr );
var
  pck: pck_ptr;

begin
  with pz^ do
  begin
    pck := pz^.pckf;
    while pck <> nil do
      with pck^ do
      begin
        if enaflg then
        begin
          thpos := savth;
          heigh := savhg;
          width := savwd
        end;
        pck := next
      end;
    for ii := 1 to max_bckparm do
    begin  bcf[ii] := bsav[ii]; bcs[ii] := -1.0  end;
    ADJUST_PEAK_POS( pz )
  end
end RELOAD_ZPCK;



[global]
procedure INTEGR$RELOAD_A_PCK;
var
  pz: zon_ptr;

begin
  pz := zone_first;
  while pz <> nil do
  begin  RELOAD_ZPCK( pz ); pz := pz^.next  end
end INTEGR$RELOAD_A_PCK;



[global]
procedure INTEGR$RELOAD_Z_PCK( th: real );
var
  pz: zon_ptr;

begin
  pz := zone_first;
  while pz <> nil do                                  { Loop to locate the zone }
    with pz^ do
    begin
    exit if (th >= thmin) and (th <= thmax);
      pz := next
    end;
  if pz <> nil then
  begin  RELOAD_ZPCK( pz );  end
end INTEGR$RELOAD_Z_PCK;



procedure SAVE_ZPCK( pz: zon_ptr );
var
  pck: pck_ptr;

begin
  with pz^ do
  begin
    pck := pckf;
    while pck <> nil do
      with pck^ do
      begin
        if enaflg then
        begin
          savth := thpos;
          savhg := heigh;
          savwd := width
        end;
        pck := next
      end;
    for ii := 1 to max_bckparm do bsav[ii] := bcf[ii]
  end
end SAVE_ZPCK;



[global]
procedure INTEGR$SAVE_A_PCK;
var
  pz: zon_ptr;

begin { INTEGR$SAVE_PEAKS }
  pz := zone_first;
  while pz <> nil do
  begin  SAVE_ZPCK( pz ); pz := pz^.next  end
end INTEGR$SAVE_A_PCK;



[global]
procedure INTEGR$SAVE_Z_PCK( th: real );
var
  pz: zon_ptr;

begin
  pz := zone_first;
  while pz <> nil do                                  { Loop to locate the zone }
    with pz^ do
    begin
    exit if (th >= thmin) and (th <= thmax);
      pz := next
    end;
  if pz <> nil then SAVE_ZPCK( pz )
end INTEGR$SAVE_Z_PCK;



[global]
function  INTEGR$SET_REFZONE( lt, rt: real ): ^string;
{ Create or Update a Theta Zone }
const
  tol = 0.25;

var
  pz, pz1, pz2:        zon_ptr;
  pk, pk1:             pck_ptr;
  ip,
  iul, iur, izl, izr:  integer;
  bcc, bfn, bok, berr: boolean;

begin
  { Find the pattern zone limit in dot index }
  iul := LOCATE_ANGLE( lt, tol, sel_ileft, true );
  iur := LOCATE_ANGLE( rt, tol, iul, true );
  berr:= false;
  bok := false;
  bcc := false;
  bfn := false;                                         { Assume zone not found until shown otherwise }
  with sel_pat^ do
  begin
    pz  := zone_first;
    while pz <> nil do                                  { Loop on all already defined zones }
    with pz^ do
    begin
      izl := lflim;  izr := rilim;                      { Get the effective founded zone limits }
    exit if iur < izl;                                  { End of zone search loop without the current zone }
      if iul <= izr then
      begin                                             { This zone are some common dots : }
        bfn := true;                                    { - Set the flag to signal a zone edition }
        if iul <> izl then                              { - We must change the zone at left limit }
        begin
          dat[iul].mflg := dat[iul].mflg + [mk_rfleft];
          dat[izl].mflg := dat[izl].mflg - [mk_rfleft];
          thmin := dat[iul+1].theta;                    { Set the new (2*)theta minimum }
          lflim := iul;  lfmar :=   1                   { Set the new left limit }
        end;
        bok := iur <= izr;                              { Flags the end of zone search loop with this zone }
        if iur <> izr then                              { - We must change the right limit }
        begin
          if iur > izr then bcc := true;                { Set the flag to concate the following zone if possible }
          dat[izr].mflg := dat[izr].mflg - [mk_rfright];
          dat[iur].mflg := dat[iur].mflg + [mk_rfright];
          thmax := dat[iur-1].theta;                    { Set the new (2*)theta maximum }
          rilim := iur; rimar := 1                      { Set the new right limit }
        end
      end;
    exit if bok or bcc;
      pz  := next
    end;

    if bfn then                                         { When it is not a new zone (pz is the zone address) }
    begin                                               { We must edit the founded zone }
      if bcc then                                       { Some zone must be concatened to the founded one }
      begin
        pz1 := pz^.next;                                { Begin of the Search Loop of a previuosly existing zone }
        while pz1 <> nil do                             { pz-> the founded zone, pz1-> a next zone }
        with pz1^ do
        begin
        exit if lflim > iur;                            { Stop loop when the begin of an existing zone is after the end of unew zone }
          pk := pz1^.pckf;                              { Get the first peak address or nil }
          if pk <> nil then                             { When some peak(s) are existing in the zone }
          begin                                         { For the Zone(s) with some defined peak(s) }
            if pz^.pckf = nil then pz^.pckf := pk       { We append the peak list to the user zone }
                              else pz^.pckl^.znxt := pk;
            pz^.pckl := pz1^.pckl;
            while pk <> nil do                          { Loop on the peak(s) of old zone }
            with pk^ do
            begin  zone := pz; pk := znxt  end;         { Attach the peak(s) to the (new) zone and ... }
            pz1^.pckf := nil; pz1^.pckl := nil          { ... clear the old zone peak list to keep them (during old zone free) }
          end;
          { Supress the graphic marker of the old zone }
          if iur <> lflim then dat[lflim].mflg := dat[lflim].mflg - [mk_rfleft];
          if iur <> rilim then dat[rilim].mflg := dat[rilim].mflg - [mk_rfright];
          pz2 :=  pz1;
          pz1 := next;
          FREE_ZONE( pz2 )                              { Free the old zone allocations }
        end                                             { End of zone concate and free loop }
      end;

      { Loop to supress any peak(s) out of new (or updated) zone (2*)theta range for any edited zone }
      with pz^ do
      begin
        pk := pckf;
        npeak := 0;
        while pk <> nil do
        with pk^ do
        begin
          pk1 := pk;
          pk  := znxt;
          if (thpos < thmin) or (thpos > thmax) then
          begin { When the Peak is out of the updated zone }
(*
;WRITELN( ' Suppress peak at ', thpos:8:3 );
*)
            ip := LOCATE_ANGLE( thpos, tol, izl, true, true );
            if (ip <= 0) or (ip > sel_iright) then      { When the peak cannot be localized => Error ! }
              berr := true
            else
              with dat[ip] do
                mflg := mflg - [mk_rflcen,mk_rflpos,mk_rflhkl];
            FREE_PEAK( pk1 )
          end
        end
      end
    end
    else
    begin                                               { Create a new independant empty zone }
      CREATE_ZONE( iul, iur, pz );
      dat[iul].mflg := dat[iul].mflg + [mk_rfleft];
      dat[iur].mflg := dat[iur].mflg + [mk_rfright]
(*
;WRITELN( ' Create a new zone in range ', iul:0, '..', iur:0, ' from ', lt:8:3, ' to ', rt:8:3 );
*)
    end
  end;

  bfitini := true;                                      { Set future init mode for Reflection Least-Squares fitting }

  if berr then INTEGR$SET_REFZONE := ndt_msg"address
          else INTEGR$SET_REFZONE := nil
end INTEGR$SET_REFZONE;



[global]
function  INTEGR$SET_REFLINE( th, hg, wd: real; fhkl: mrk_flgtyp := [] ): ^string;
{ Create a new peak at angle th, with the height hg and the half width wd }
const
  tol     = 0.25;

  nzn_msg = '*** RPW Error: A reflection line insertion cannot be out of any integration zone.';

var
  pz:                 zon_ptr;
  pk:                 pck_ptr;
  ip, ib, sz:         integer;
  bk, x1, x2, y1, y2:    real;
  pmsg:               ^string;

begin
  { Find the pattern zone limit in dot index }
  pmsg := nil;                                          { Assume no error until shown otherwise }
  if (sel_pat <> nil) and (bckgrd_crv <> nil) then      { Possible when a pattern is loaded and the background is created }
  with sel_pat^ do
  begin
    pz := zone_first;
    while pz <> nil do                                  { Loop on the zone(s) to find the peak zone owner }
    begin
    exit if (th >= dat[pz^.lflim].theta) and (th <= dat[pz^.rilim].theta);
      pz := pz^.next
    end;
    if pz = nil then goto E_END;                        { Nothing to do when the theta is not inside a zone }

    with pz^ do
    begin
      ip := LOCATE_ANGLE( th, tol, lflim, true, true ); { Locate the pattern point in the zone }
      if (ip <= 0) or (ip > rilim) then
      begin  pmsg := ndt_msg"address; goto E_END end    { Array index error: angle is not found in the tolerance }
    end;

    with dat[ip], bckgrd_crv^ do
    begin                                               { We have find the zone to insert a new peak }
      pk := CREATE_PEAK( pz, th, wd );                  { Create the reflection peak record }
      sz := crv_sz*2; ib := 2;                          { Get the background table size }
      x2 := crv_tab[1]; y2 := crv_tab[2];
      x1 := x2 - 1.0; y2 := y1;                         { To force a default cte }
      while (ib <= sz) and (x2 < th) do                 { Loop to find two points of curve ... }
      begin                                             { ... x1 <= th <= x2 }
        x1 := x2; y1 := y2;
        ib := ib + 1; x2 := crv_tab[ib];
        ib := ib + 1; y2 := crv_tab[ib]
      end;
      if hg <= 0.0 then hg := int;                      { By default the high of dot is the experimental ordinate }
      {ca := (y2 - y1)/(x2 - x1); cb := y2 - ca*x2;}
      bk := (y2 - y1)*(th - x2)/(x2 - x1) + y2;
      pk^.heigh :=   hg - bk;                           { Set the reflection high }
      pk^.savhg := pk^.heigh;
      if fhkl <> [] then mflg := mflg + fhkl*[mk_rflpos,mk_rflcen,mk_rflhkl]
                    else mflg := mflg + [mk_rflcen];    { Set the graphic mark }
      bfitini := true                                   { Set future init mode for Reflection Least-Squares fitting }
    end
  end;
E_END:
  INTEGR$SET_REFLINE := nil
end INTEGR$SET_REFLINE;



[global]
function  INTEGR$ZONE_DEL( th: real ): ^string;
{ Suppress an angular integration zone }
const
  tol     = 0.25;

  nzn_msg = '*** RPW Error: Cannot find the integration zone to suppress.';

var
  pz:          zon_ptr;
  pk:          pck_ptr;
  ip:          integer;
  pmsg:        ^string;

begin
  { Find the pattern zone limit in dot index }
  if (sel_pat <> nil) and (zone_first <> nil) then
  with sel_pat^ do
  begin
    pz := zone_first;
    while pz <> nil do                                  { Loop to locate the zone }
    with pz^ do
    begin
    exit if (th >= thmin) and (th <= thmax);
      pz := next
    end;
    pmsg := nzn_msg"address;
    if pz = nil then goto E_END;                        { Not such specified zone error }
    pk := pz^.pckf;
    while pk <> nil do
    with pk^ do
    begin
      ip := LOCATE_ANGLE( thpos, tol, sel_ileft, true, true );
      pmsg := ndt_msg"address;
      if ip = 0 then goto E_END;                        { Cannot find the related pattern dot }
      with dat[ip] do  mflg:=mflg-[mk_rflcen,mk_rflpos,mk_rflhkl];  { Clear the reflection mark }
      pk := znxt
    end;
    with dat[pz^.lflim] do mflg := mflg - [mk_rfleft];  { Delete the zone limit markers }
    with dat[pz^.rilim] do mflg := mflg - [mk_rfright];
    FREE_ZONE( pz );                                    { Free the zone and all its included peaks }
    pmsg := nil
  end;
E_END:
  INTEGR$ZONE_DEL := pmsg
end INTEGR$ZONE_DEL;



[global]
function  INTEGR$FREE_ALLZ: ^string;
begin
  if zone_first <> nil then
  begin
    if sel_pat <> nil then
    with sel_pat^ do
      for ii := sel_ileft to sel_iright do      { Loop on the area pattern points ... }
        with dat[ii] do                         { ... to delete all zone and reflection markers }
          mflg := mflg - [mk_rfleft,mk_rfright,mk_rflcen,mk_rflpos,mk_rflhkl];
    FREE_ALL_ZONE
  end;
  INTEGR$FREE_ALLZ := nil
end INTEGR$FREE_ALLZ;



[global]
function  INTEGR$LINE_DEL( th: real ): ^string;
{ Supress a diffraction line }
const
  tolu    =  0.50;                                      { 2*theta tolerance for user }
  told    =  0.25;                                      { 2*theta tolerance for user }

  npk_msg = '*** RPW Error: No Reflection Lines find in the 2*theta tolerance (0.25�)';

var
  pk, pk1:            pck_ptr;
  thc, thm:              real;
  ip:                 integer;
  pmsg:               ^string;

begin
  { Find the pattern zone limit in dot index }
  pmsg := nil;                                          { Assume no error until shown otherwise }
  if (sel_pat <> nil) and (peak_first <> nil) then
  with sel_pat^ do
  begin
    pmsg := npk_msg"address;
    pk1  :=      peak_first;
    pk   :=             nil;
    thm  :=           360.0;
    while pk1 <> nil do                                 { Loop to get the nearest peak of the provided theta }
    with pk1^ do
    begin
      thc := ABS( thpos - th );
(*
WRITELN( ' Ref search for ', th:8:2, ' find peak at ', thpos:8:2, ' del = ', thc:8:3, ' thm = ', thm:8:3 );
*)
    exit if thc > thm;                                  { Exit of Loop when current peak too right from the search angle }
      thm := thc; pk := pk1;
      pk1 := next
    end;
    if (pk = nil) or (thm > tolu) then goto E_END;      { Cannot find a good agreement for a line }
    with pk^ do
    begin
      ip := LOCATE_ANGLE( thpos, told, sel_ileft, true, true ); { Get the reflection peak position index }
      pmsg := ndt_msg"address;

      if ip = 0 then goto E_END;                        { Cannot find the related pattern dot }
      pmsg := nil;
      with dat[ip] do
        mflg := mflg - [mk_rflcen,mk_rflpos,mk_rflhkl];
    end;
    FREE_PEAK( pk )
  end;
E_END:
  INTEGR$LINE_DEL := pmsg
end INTEGR$LINE_DEL;



end RPW_INT_SEARCH.
