{
*************************************************************************
*                                                                       *
*                                                                       *
*        R P W D A T A  (Reduce Powder DATA for Diffraction)            *
*                                                                       *
*                2D DRAW Interface Graphic Module                       *
*                                                                       *
*             ( ILL Data Base Manager Source File )                     *
*                                                                       *
*                Version  1.1-C  - - 30-Nov-2009                        *
*                                                                       *
*                                by                                     *
*                                                                       *
*                  Pierre Wolfers, Institut Neel                        *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
*************************************************************************

/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////

}
module RPW_INT_COMPUTE;

%include 'RPWSRC:rpwdata_int_env.pas' {, list_on};



[global]
var
  width_free:          boolean;                 { Key for the width }



procedure DISPLAY_FITDOT( x, y: lsq_flt );
const
  cvrseg = 240;

begin
  if fitcmp_crv = nil then
    { NEW_CURVE( size, ide, line_color, marker_color, line_kind, marker_kind, line color, marker_color ) }
    fitcmp_crv := NEW_CURVE( cvrseg, 2, 3, 0, 1, 0, 2.0, 0.0 );

  if fitcmp_cnp >= fitcmp_crv^.crv_sz then UPDATE_CURVE( fitcmp_crv, - (fitcmp_crv^.crv_sz + cvrseg) );

  with fitcmp_crv^ do
  begin
    crv_tab[2*fitcmp_cnp+1] := x;
    crv_tab[2*fitcmp_cnp+2] := y + cshift;
    fitcmp_cnp := fitcmp_cnp + 1
(*
;WRITELN( ' FIT Curve dot # ', fitcmp_cnp:5, x, y )
*)
  end
end DISPLAY_FITDOT;



procedure FPCK_DEFINE( pz: zon_ptr; bpck: boolean; dmp, mrq: lsq_flt; bth: boolean := true );
{ Define all LSQ Variables for the zone pz to fit the following parameters :
   if bpck then
                the peaks of the pz^ zone will be fitted with background parameters,
           else
                only the bacground parameters will be fitted.
   if bth then (used with bpck = true)
                the peaks positions  will be locked (used for first cycles.
           else
                the peaks positions  will be fitted with the other parameters.

      fbk_n is the polynome order to fit the background.

      bkm_i is 0,1 or 2 to do respectively :   Use initial background (GBK), used first fitted background (FBK),
                                               fit background with the peaks (FFBK).
                => The background is fitted when bkm_i = 2 or bpck = false (no peak fit).
           else the initial background is the (Hystogram or entered background).

      rwm_i is 0, 1 or 2 to do respectively :  Fit of individual half width (one parameter for each peak),
                                               Fit a common half width for all peaks of the zone,
                                               The half width for all peaks are given by the formulae w0 + w1*Tang( 2*theta )
                                               where w0 and w1 are two fitted parameters.

      rps_i is 0 or 1 to fit individual or zone common profil mixing parameter for pseudo-voigt profile.

   Note dmp and mrq are respectively the damping and Marqward Levenberg factors.
}
var
  ior: integer;
  pk:  pck_ptr;

begin
  with hkl_parm, pz^ do
  begin
    LSQ$INIT;                                           { Init the least-squares engine }
    idgw :=  0;
    idsp :=  0;
    ior  :=  0;

    if (not bpck) or (bkm_i = 2) then                   { When the background is fitted ... }
      for i := 1 to fbk_n + 1 do                        { ... we start from the previous fitted values }
        ior := LSQ$NEW_VPARM( bcf[i], bcs[i] );         { We define the background parameter to fit }

    if bpck then
    begin
      if (rwm_i > 0) and pwd_f then
      begin
        ior  := LSQ$NEW_VPARM( zwid0, zswi0 );          { We define the global main width to fit }
        idgw := ior;                                    { Keep the width index }
        if rwm_i > 1 then
          ior := LSQ$NEW_VPARM( zwid1, zswi1 )          { Set the fit of TAN term }
      end;

      if (rps_i = 1) and pmx_f then
      begin
        ior  := LSQ$NEW_VPARM( zprf0, zspr0, 0.0,1.0 ); { We define the global first profil special parm }
        idsp := ior                                     { Can have second profil parm. (in future) depending of profil type }
      end;

      { Scan all peak of zone }
      pk := pckf;
      while pk <> nil do
        with pk^ do                                     { Loop for each peak in the zone (packet) }
        begin
          idthe := 0;
          idhig := 0;
          idpr0 := 0;
          idpr1 := 0;
          idwid := 0;

          if bth and pth_f then
            idthe := LSQ$NEW_VPARM( thpos, sthpo, thmin, thmax );       { Fit the reflection position }

          if phg_f then
            idhig := LSQ$NEW_VPARM(  heigh, shigh, 0.0, flt_max );      { Fit packet high }

          if pwd_f then
            if rwm_i <= 0 then idwid := LSQ$NEW_VPARM( width, swidt )
                          else idwid := idgw + 1;
          if (* (rps_i <= 0) and *) pmx_f then
          case ftyp of
            fnc_pvoigt:                                  { For Pseudo-Voigt we add the mixing parameter }
              if rps_i <= 0 then idpr0 := LSQ$NEW_VPARM( mixfc, smixf, 0.0, 1.0 )
                            else idpr0 := idsp;

          otherwise
          end;
          pk := znxt
        end
    end;
    LSQ$NEW_BLOCK( dmp, mrq )                           { Create the matrix and vectors for one unique diagonal block }
  end
end FPCK_DEFINE;



procedure FPCK_CYCLE( pz: zon_ptr; bnd, bpck: boolean; var nobs: integer; var bchg: boolean );
{ Performs one cycle the Least-Squares (LSQ) on the ps^ zone.

  bnd     is used to perform a final cycle without LSQ Variable change,
  bpck    is usec to enable the peak parameters Fit.

  nobs will be the returned number of used observation point and bchg .
}
var
  lchi2, bk, sg, lpf2, mt, rv, th, tgthp, we, nchf:  lsq_flt;
  ib, im, ip, it, sz, ndat:                          integer;
  pk:                                                pck_ptr;
  xx, yy:                             array[1..2] of lsq_flt;


  function CHPCK_COEF( i: integer; v_o, v_n, v_c, v_s, min, max: lsq_flt ): lsq_flt;
                      { pindex - old - new - change - sigma }
  var
    blm: boolean := true;

  begin
    { Check for peak limit angles move inside the zone }

   if (v_n < min) then v_n := min
                  else if (v_n > max) then v_n := max
                                      else blm := false;

    { Output paralmeter Change when Required }
    if pro_parm.idb_f then
    begin
      WRITE( ' P # ', i:3, ' : new = ', v_n, ', sg = ', v_s, ', old = ', v_o, ', change = ', v_c );
      if blm then WRITELN( ' Lm')
             else WRITELN
    end;
    CHPCK_COEF := v_n
  end CHPCK_COEF;



  function  REF_DCOMPUTE( pk: pck_ptr; th: lsq_flt ): lsq_flt;
  { Compute the profile function and put the partial derivates in the sigma }
  var
    dh, ew, f1, f2, re: lsq_flt;

  begin
    with hkl_parm, pk^ do
    begin
      dh := th - thpos;
      if width < 1.0e-5 then width := 1.0e-5;
      case ftyp of
        fnc_gauss:
          begin
            ew    := width*gsc1;
            f1    := SQR( dh/ew );
            shigh := EXP( - f1 );
            re    := heigh*shigh;
            sthpo := 2.0*(dh/SQR( ew ))*re;
            swidt := 2.0*(f1/width)*re
          end;

        fnc_lorentz:
          begin
            shigh := 1.0/(1.0 + SQR( dh/width ));
            re    := heigh*shigh;
            sthpo := 2.0*dh*heigh*SQR( shigh/width );
            swidt := sthpo/width
          end;

        fnc_pvoigt:
          begin
            ew    := width*gsc1;
            f1    := EXP( - SQR( dh/ew ) );
            f2    := 1.0/(1.0 + SQR( dh/width ));
            shigh := (1.0 - mixfc)*f1 + mixfc*f2;
            re    := heigh*shigh;
            smixf := heigh*(f2 - f1);
            sthpo := 2.0*heigh*((dh/SQR( ew ))*(1.0 - mixfc)*f1 + mixfc*SQR( f2/width ));
            swidt := sthpo/width
          end;

      otherwise
      end
    end;
    REF_DCOMPUTE := re
  end  REF_DCOMPUTE;



begin { FPCK_CYCLE }
  if bnd then LSQ$SET_DMP_MRQ( 1, 1.0, 1.0 );                   { Reset dmp and mrq to one for final cycle }
  LSQ$INIT_CYCLE;                                               { Initialize the LSQ cycle }
  with pro_parm, hkl_parm, pz^, sel_pat^, bckgrd_crv^ do
  begin
    if (bkm_i = 0) and bpck then                                { We use the GBK mode (initial) background with peak fit }
    begin
      sz := 2*crv_sz;
      ib := 4;
      xx[1] := crv_tab[ib-3]; yy[1] := crv_tab[ib-2] - cshift;
      xx[2] := crv_tab[ib-1]; yy[2] := crv_tab[ib]   - cshift;
      bcf[2] := (yy[2] - yy[1])/(xx[2] - xx[1]);                { Get the first background line equation }
      bcf[1] := yy[2] - bcf[2]*xx[2]
    end;
    mt := (dat[lflim].theta + dat[rilim].theta)*0.5;            { Get the abcisse of the zone middle }

    ndat :=     0;
    it   := lflim;                                              { Loop one all pattern zone points }
    repeat
      with dat[it] do
      if not (mk_invalid in mflg) then                          { Skip any invalid pattern dot }
      begin
        LSQ$INIT_DERV;                                          { Init the table partial derivates (create it is not existing) }
        { Compute the background and set its related derivate when it is fitted (mode FFBK) }
        if bpck then im := bkm_i
                else im :=     2;                               { Force refinement of background when no refl. fit }
        case im of
          1: begin      { FBK mode already fitted }
              th := theta - mt; rv := th;
              bk := bcf[1]; sg := SQR( bcs[1] );
              for i := 2 to fbk_n + 1 do
              begin
                bk := bk + bcf[i]*rv; rv := rv*th;
                sg := sg + SQR( bcs[i]*rv )                     { Warning! the Correlation coefficients was forgotten!!! }
              end
            end;

          2: begin      { FFBK mode : we fit the background with the peaks }
              th := theta - mt;
              bk := 0.0; rv := 1.0;
              for i := 1 to fbk_n + 1 do
              begin
                bk := bk + bcf[i]*rv;                           { Compute the background value }
                LSQ$SET_DERV( i, rv );                          { Set the ith partial derivate of background }
                rv := rv*th                                     { Update the power of theta (for bck, org at zone middle) }
              end;
              sg := 0.0                                         { No background sigma in FFBK mode }
            end;

        otherwise       { GBK mode }
          if ib < sz-2 then
          begin
            while (ib < sz-2) and (ib > 2) and (crv_tab[ib-1] <= theta) do ib := ib + 2;
            xx[1] := crv_tab[ib-3]; yy[1] := crv_tab[ib-2] - cshift;    { Get the best GBK point for the current (2*)theta }
            xx[2] := crv_tab[ib-1]; yy[2] := crv_tab[ib]   - cshift;
            bcf[2] := (yy[2] - yy[1])/(xx[2] - xx[1]);          { Get the first background line equation }
            bcf[1] := yy[2] - bcf[2]*xx[2];
            sg := (SQR( dat[lflim].sig ) + SQR( dat[rilim].sig ))*0.5;  { Set the background squared sigma }
            bk := bcf[1] + bcf[2]*theta                         { Get background value for the dot }
          end
        end;

        lpf2 := bk;
        { Scan on all peaks in the 2*theta zone when required }
        if bpck then
        begin                                                   { When a peak fit is required }
          pk := pckf;
          while pk <> nil do                                    { Loop on all peak }
          with pk^ do
          begin
            if rwm_i > 0 then
            begin
              tgthp := TAN( th2brr*thpos );                     { Get the TAN( 2�theta ) value for derivate }
              width := zwid0 + zwid1*tgthp                      { Set the new value of peak width }
            end;
            if rps_i > 0 then  mixfc := zprf0;                  { Set the special parameter when they are global }

            lpf2 := lpf2 + REF_DCOMPUTE( pk, theta );           { Compute the peak contribution and derivates (in sigma place) }
(*
WRITELN( ' # ', idthe:3, ' derv theta = ', sgthp );
WRITELN( ' # ', idhig:3, ' derv high  = ', sghig );
WRITELN( ' # ', idwid:3, ' derv width = ', sgwid );
*)
            if idthe > 0 then LSQ$SET_DERV( idthe, sthpo );     { Set the 2*theta position derivation }
            if idhig > 0 then LSQ$SET_DERV( idhig, shigh );     { Set the peak high derivation }

            if rwm_i <= 0 then                                  { For individual width ... }
            begin
              if idwid > 0 then LSQ$SET_DERV( idwid, swidt )    { Set the local width derivation when it is fitted }
            end
            else
            begin
              LSQ$SET_DERV( idgw, swidt );                      { Set the global derivation and, when required ... }
              if rwm_i > 1 then LSQ$SET_DERV( idgw+1, swidt*tgthp )     { ... with TAN(thpos) term }
            end;

            { When some special profil parm. are used }
            if idpr0 > 0 then LSQ$SET_DERV( idpr0, smixf );

            pk := znxt                                          { Skip to next peak }
          end
        end;

        { The old sg is the sigma^2 of background when it is not fitted }
        if sig < 1e-4 then sg := SQRT( ABS( int ) )
                      else sg := sig;
        if sg < 1e-4 then sg := 1e-4;
        LSQ$SET_CONTR( int - lpf2, 1.0/sg );
(*
WRITELN( it:5, '/ ', theta:8:3, ' O = ', int:10:1, ', C = ', lpf2:10:1, ', sg = ', sg:9:1, ', R = ', (int - lpf2)/sg:10:3 );
*)

        if bnd and bpck and ddy_f then DISPLAY_FITDOT( theta, lpf2 );

        ndat := ndat + 1                                { Count of used data dots }
      end;

      it := it + 1;
      if not bpck then
        if it = lflim + lfmar then it := rilim - rimar + 1
    until it > rilim
  end;
  bchg := LSQ$PROCESS( bnd, not bnd, bpck, CHPCK_COEF );{ Complete the Least Squares cycle }
  nobs := ndat
end FPCK_CYCLE;



procedure FPCK_MRQFIT( pz: zon_ptr; bend: boolean := true );
{ Performs a complete fit for the zone pz^.

  bend is used to require a final cycle on the zone pz^.
}
const
  eps = 1.0e-4;        { Epsilon for convergence test }
  mfc =   10.0;        { Levenberg-Marquardt change factor }

var
  ic, ik0, ik1, ndata: integer;
  pmrq, lmrq, mcoef:   lsq_flt;
  bch, bok:            boolean;

begin
  with hkl_parm, pro_parm do
  begin
    bok   :=    true;
    mcoef :=     1.0;
    lmrq  :=     0.5;
    ic := 0; ik0 := 0; ik1 := 0;
    repeat
      ic := ic + 1;
      pmrq := lmrq;
      lmrq := 1.0/(1.0 + mcoef);
      LSQ$SET_DMP_MRQ( 1, {dmp_r} lmrq, lmrq );
      if idb_f then WRITELN( ' Start Cycle # ', ic:3, ' with mrq = ', lmrq:8:4 );
      FPCK_CYCLE( pz, false, true, ndata, bch );
      if not bch then
      begin
        ik1 := ik1 + 1;
        mcoef := mcoef*mfc;
(*
WRITELN( ' Bad cycle change mcoef to ', mcoef );
*)
        if idb_f then
          WRITELN( ' Cycle # ', ic:3, ' Canceled (Too large resulting chi2 : ', chi2:10:2, ' )' );
        LSQ$RESTORE_PARM;
        FPCK_CYCLE( pz, false, true, ndata, bch );
        ik0 := 0; bok := false
      end
      else
      begin
        if ABS( lchi2 - chi2 ) <= chi2eps then ik0 := ik0 + 1
                                          else ik0 := 0;
        if bok then mcoef := mcoef/mfc;
(*
WRITELN( ' Good cycle with mcoef = ', mcoef );
*)
        ik1 := 0;
        if idb_f then
          WRITELN( ' Cycle # ', ic:3, ' with mrq = ', lmrq:8:4, ' : Chi2 = ', chi2:10:2 );
        bok := true
      end;

    { if (chi2 < lchi2) then lchi2 := chi2; }

    until (ic >= cym_n) or (ik0 > 1);

    LSQ$SET_DMP_MRQ( 1, 1.0, 1.0 );
    if bend then FPCK_CYCLE( pz,  true, true, ndata, bch )           { Perform the final cycle }
  end
end FPCK_MRQFIT;



procedure FIT_PACKET( pz: zon_ptr );
{ Performs a complete refinment a the pz^ zone.


}
var
  ic, ip:                         integer;
  pk:                             pck_ptr;
  lchi2, lsq_dmp, lsq_mrq: lsq_flt := 1.0;

label
  ET_ONERROR;                  { Label to return on Numeric error }



  function NUM_ERROR_HDL( ierr: cc__int ): cc__int;

    procedure GEN_ERROR_MSG( ierr: cc__int ); external 'PAS__GEN_ERROR_MSG';

  begin
    if(ierr >= 20) and (ierr <= 29) then
    begin
      WRITELN;
      WRITELN( ' *** Numeric error during the RPW Integration/Fit process ***' );
      WRITELN;
      GEN_ERROR_MSG( ierr );   { Generate the standard RTL message }
      WRITELN;
      nerr_flg := true;        { Flag the numeric error }

      goto ET_ONERROR          { Return from Fit process }
    end;
    NUM_ERROR_HDL := -1        { Standard action for the other errors }
  end NUM_ERROR_HDL;




  procedure REF_ICOMPUTE( pk: pck_ptr );
  { Compute the intensity and sigma :
    The ordering of LSQ parameters is always thpos, high, width [, oxp1 [, oxp2 ]].
    Consequently, the correlation matrix element C(high,width) can be indexed by [ior+2,ior+3].
  }
  const
    c1 = gsin - pi;

  var
    f1, f2:    real;

  begin
    with pk^ do
    begin
      case ftyp of
        fnc_gauss:
          begin
            intens := gsin*heigh*width;
            if (shigh >= 0.0) and (swidt >= 0.0) then
              sigma  := gsin*SQRT( SQR( shigh*width ) + SQR( swidt*heigh ) +
                                   shigh*swidt*heigh*width*LSQ$CORREL( idhig, idwid ) )
            else sigma := -1.0
          end;

        fnc_lorentz:
          begin
            intens := pi*heigh*width;
            if (shigh >= 0.0) and (swidt >= 0.0) then
              sigma  := pi*SQRT( SQR( shigh*width ) + SQR( swidt*heigh ) +
                                 shigh*swidt*heigh*width*LSQ$CORREL( idhig, idwid ) )
            else sigma := -1.0
          end;

        fnc_pvoigt:
          begin
            intens := (mixfc*gsin + pi*(1.0 - mixfc))*heigh*width;
            if (shigh >= 0.0) and (swidt >= 0.0) and (smixf >= 0.0) then
            begin
              f1     := (1.0 - mixfc)*gsin + mixfc*pi;
              f2     := f1*c1;
              sigma  := SQRT( SQR( smixf*c1*heigh*width ) +
                              SQR( shigh*f1*width ) + SQR( swidt*f1*heigh ) +
                              SQR( f1 )*heigh*width*shigh*swidt*LSQ$CORREL( idhig, idwid ) +
                              f2*heigh*SQR( width )*smixf*shigh*LSQ$CORREL( idhig, idpr0 ) +
                              f2*SQR( heigh )*width*smixf*swidt*LSQ$CORREL( idwid, idpr0 ) )
            end
            else sigma := -1.0
          end;

      otherwise
      end
    end
  end REF_ICOMPUTE;



begin { FIT_PACKET }
  ESTABLISH( NUM_ERROR_HDL );                           { Enable the Numeric error handler }
  if pz <> nil then                                     { Loop on each zone }
  with hkl_parm, pro_parm, pz^ do
  begin

    if bfitini then
    begin
      FPCK_DEFINE( pz, true, lsq_dmp, lsq_mrq, false ); { Fit with theta locked in Least-Squares base }
      FPCK_MRQFIT( pz, false );
      INTEGR$SAVE_A_PCK                                 { Save the state of all peak(s) and all zone(s) }
    end;

    FPCK_DEFINE( pz, true, lsq_dmp, lsq_mrq, true );    { Fit all parameters to fit in Least-Squares base }
    FPCK_MRQFIT( pz );

    if not bprocess then ADJUST_PEAK_POS( pz );         { Shift the Peak marker to there new positions }

    rchi2 := chi2;                                      { Save the resulting goodness of fit }
    WRITELN( ' Goodness of Fit (Chi2) = ', chi2:10:4 );
    pk := pckf;
    while pk <> nil do
      with pk^ do                                       { Loop on all peak of the zone }
      begin
        REF_ICOMPUTE( pk );                             { Compute the reflection intensity }
        if idb_f then WRITELN( ' Fraction of background = ', width:8:3, ' / ', zwidth:8:3 );
        if (bkm_i < 2) and (sigma >= 0.0) then sigma := SQRT( SQR( sigma ) + SQR( zbks*width/zwidth ) );

        WRITELN( ' *** Peak at ', thpos:8:3, ':', sthpo:8:3, ', I = ', intens:10:2, ':', sigma:10:2,
                                  ', H = ', heigh:10:1, ':', shigh:9:1, ', W = ', width:8:3, ':', swidt:8:3 );

        pk := znxt                                      { Skip to next peak }
      end;
    WRITELN;
    WRITELN
  end;

ET_ONERROR:                                             { Stop computing on numeric error during the fit }
  REVERT                                                { Disable the numeroc error Handler }
end FIT_PACKET;




procedure INTEGR_PACKET;
const
  lsq_dmp     =    1.0;
  lsq_mrq     =    1.0;

var
  pz:                                  zon_ptr;
  pk:                                  pck_ptr;
  bch:                                 boolean;

  bk, sg, va, vb, lv, rv,
  lw, rw, dt, mt, ob, os,
  pb, po, pt, th, lt, rt:                 real;
  lr, rr, ib, ip, sz, nz, ndata:       integer;

  xx, yy:          array[1..2] of real;
  bcd:   array[1..max_bckparm] of real;     { Derivate value table for Background Least-Squares (FBK mode) }


begin { INTEGR_PACKET }
  with hkl_parm, pro_parm, sel_pat^, bckgrd_crv^ do
  begin
    width_free := (wd0_f or wd1_f or wd2_f);            { Get the global Width free key }
    nz :=          0;
    pz := zone_first;
    while (pz <> nil) and (nz < nzone) do               { Loop on all zone }
    with pz^ do
    begin                                               { For this (2*)theta zone }
      nz := nz + 1;                                     { Set the zone number }
      pk  := pckf;                                      { Get the first peak, record pointer }
      lr  := lflim + lfmar; rr  := rilim - rimar;       { Get the reflection line zone limits }
      lt  := dat[lr].theta; rt  := dat[rr].theta;       { Get the theta limits }
      mt  := (rt + lt)/2.0;                             { Get the middle point of region }
      lt  :=       lt - mt; rt  :=       rt - mt;       { Get with local origine }
      dt  := rt - lt;                                   { Get the width of scan in 2*theta units }
      zwidth  :=  dt;                                   { Save it in the zone record }
      if (bkm_i = 0) or (fbk_n < 0) then                { We use the hystogram_generated or entered Background (=GBK) }
      begin
        sz := 2*crv_sz;
        ib := 4;
        { Loop to search a first Initial (Hystogram) Background point: 1 before, 1 after the limit if possible }
        while (ib < sz) and (crv_tab[ib-1] <= dat[lr].theta) do ib := ib + 2;
        xx[1] := crv_tab[ib-3]; yy[1] := crv_tab[ib-2] - cshift;
        xx[2] := crv_tab[ib-1]; yy[2] := crv_tab[ib]   - cshift;
        bcf[2] := (yy[2] - yy[1])/(xx[2] - xx[1]);      { Get the first background line equation }
        bcf[1] := yy[2] - bcf[2]*xx[2];
        sg := (SQR( dat[lr].sig ) + SQR( dat[rr].sig  ))*0.5;   { Set the background squared sigma }
        pb := bcf[1] + bcf[2]*dat[lr].theta;            { Keep the initial value of background }
        bk := 0.0                                       { Init the background for the summ in the scan of zone dots }
      end
      else                                              { We use a fitted background ( = FBK) that is }
      begin                                             { ... a polynome or 0, 1, 2, or 3th degree (for fbk_n in 1..4). }
        for i := 1 to fbk_n + 1 do                      { Loop to init the background  parameters }
        begin  bcf[i] := 0.0; bcs[i] := 0.0  end;
        FPCK_DEFINE( pz, false, 1.0, 1.0 );             { Define the background Least-Squares parameters to fit }
        FPCK_CYCLE( pz, false, false, ndata, bch );     { Perform one cycle to compute the background parameters ... }
        FPCK_CYCLE( pz,  true, false, ndata, bch );     { ... and a final cycle to get the correlation factors }
        bchi2 := chi2;                                  { Save the background fit chi2 }

        sg := 0.0;
        va := 1.0;
        bk := 0.0;
        lv := 1.0;
        rv := 1.0;
        for i := 1 to np do                             { First loop on the np fitted parameters }
        begin
          lv := lv*lt; rv := rv*rt;                     { Build lt^i and rt^i }
          va := (rv - lv)/i;                            { Forms va = delta(th^i)/i == d(bck)/d(p(i)) }
          bk := bk + va*bcf[i];                         { Forms bk = summ(i) of p(i)*[d(bck)/d(p(i))] == background }
          lw := 1.0; rw := 1.0;
          for j := 1 to np do                           { Second loop on the np fitted parameters }
          begin
            lw := lw*lt; rw := rw*rt;                   { Build lt^j and rt^j }
            vb := (rw - lw)/j;                          { Forms vb = delta(th^j)/j == d(bck)/d(p(j)) }
            { Add terms Cij*sig(p(i))*sig(p(j))*[d(bck)/d(p(i))]*[d(bck)/d(p(j))] }
            sg := sg + LSQ$CORREL( i, j )*va*vb*bcs[i]*bcs[j]
          end
        end;
        if idb_f then
        begin
          WRITELN ( ' Background Chi2 = ', SQRT( chi2 ):10:3 );
          for i := 1 to fbk_n+1 do WRITELN( ' Background Coef', i:0, ' = ', bcf[i]:10:4, ':', bcs[i]:8:4 );
        end
      end;

      { Make the numeric integration }
      ob := 0.0; os := 0.0;
      pt := dat[lr].theta;                              { Keep the initial theta for trapeze integration }
      po :=   dat[lr].int;                              { Keep the initial value of count for trapeze integration }
      for it := lr to rr do
      with dat[it] do
        if not (mk_invalid in mflg) then                { We exclude any invalid dots }
        begin
          th  := theta - mt;                            { Get the distance between the current dot and the middle of zone }
          dt  := theta - pt;                            { Get the distance between two successive dots (in 2*theta) }
          pt  :=      theta;                            { Keep the theta as previous one }
          if fbk_n < 0 then                             { Use of GBK }
          begin { When we use the Initial Background curve to evaluate the background to integrate }
            if (theta > xx[2]) and (ib < sz) then
            begin
              xx[1] := xx[2]; yy[1] := yy[2];           { We get a new background equation }
              ib := ib + 1; xx[2] := crv_tab[ib];
              ib := ib + 1; yy[2] := crv_tab[ib] - cshift;
              bcf[2] := (yy[2] - yy[1])/(xx[2] - xx[1]);
              bcf[1] := yy[2] - bcf[2]*xx[2]
            end;
            bk := bk + 0.5*pb*dt;                       { Add the previous point contribution for background }
            pb := bcf[1] + bcf[2]*theta;                { Keep the previous background value }
            bk := bk + 0.5*pb*dt                        { Compute the total background (Trapeze integral) }
          end
          else                                          { Set the background of the dot in FBK mode }
          begin { When we work with a fitted background polynome }
            rv :=     th;
            pb := bcf[1];
            for i := 2 to fbk_n + 1 do
            begin  pb := pb + bcf[i]*rv; rv := rv*th  end
          end;

          { For FBK mode, We use the previously computed background in bk }
          ob := ob + 0.5*po*dt;                         { Add the previous point contribution for intensity scan }
          po := int;                                    { Keep the previous count value }
          ob := ob + 0.5*po*dt;                         { Compute the total intensity scan surface (Trapeze integral) }
          os := os + SQR( sig );                        { Compute the total intensity scan sigma }

          if (mflg*[mk_rflpos,mk_rflcen,mk_rflhkl,mk_rfright] <> []) and (pk <> nil) then
          with pk^ do
          begin
            heigh := int - pb;  savhg := heigh;         { Set the reflection hight, and save it }
            pk   := znxt
          end
        end;

      zbkv := bk; zbks := SQRT( ABS( sg ) );            { Save the backgound value and squared sigma }

      ob := ob - bk;                                    { Substract the background to get the reflection packet ... }
      os := SQRT( ABS( os + sg ) );                     { ... intensity and related sigma }

      ztint := ob; ztsint := os;                        { Save the packet intensity result in the zone record }

      WRITELN;
      WRITELN( ' Total Intensity for the zone # ', nz:3, ' from ', dat[lflim].theta:8:3, ' to ', dat[rilim].theta:8:3 );
      WRITELN( ' I = ', ob:10:1, ' : ', os:8:1, ' obtain with ob = ',
               ob+bk:10:1, ', bk = ', bk:10:1, ' and ', ndata:6, ' BCK dots.' );

      if idb_f then
      begin
        pk := pckf;
        while pk <> nil do
        with pk^ do
        begin
          WRITELN( ' Peak at ', thpos:8:3, ' with h = ', heigh:10:1, ', w = ', width:8:3 );
          pk := znxt
        end;
        WRITELN
      end;

      FIT_PACKET( pz );

      WRITELN;
      WRITELN;

      pz := next                                        { Skip to next zone }
    end;

    if ddy_f then
    begin
      UPDATE_CURVE( fitcmp_crv, -fitcmp_cnp );          { Suppress unused dots }
      fitcmp_cnp := 0                                   { Prepare for the next replot }
    end;

  end;
  bfitini := false                                      { Cancel the init-fit mode flag }
end INTEGR_PACKET;



[global]
function  INTEGR$REF_INTEGR: ^string;
{ Perform Integration/Evaluation of the diffraction peak intensities }
begin
(*
WRITELN( ' INTEGRATION  Req : ', sel_pat <> nil, ' ', zone_first <> nil, ' ', bckgrd_crv <> nil, ' ',
                                 peak_first <> nil, ' ', sel_iright > sel_ileft, ' ', ntotpck:0 );
*)
  str_msg.length := 0;
(*
WRITELN( ' Conditions : ', sel_pat <> nil, ' ', zone_first <> nil, ' ', bckgrd_crv <> nil,
                      ' ', peak_first <> nil, ' ', sel_iright:0, ' > ', sel_ileft:0, ' ', ntotpck:0, ' > 0' );
*)
  if (sel_pat <> nil) and (zone_first <> nil) and (bckgrd_crv <> nil) and
     (peak_first <> nil) and (sel_iright > sel_ileft) and (ntotpck > 0) then
  { When it is possible }
  begin
(*
    if pro_parm.fmd_i >= 0 then
      fitflg.iv := pro_parm.fmd_i               { Set the fit flags when specified ... }
    else fitflg.sv := [fflg_automd];            { ... else set the automatic mode }
*)

    INTEGR_PACKET;                              { Proceed to the integration by packet }

  end;
  SORT_PEAK;                                    { Make a sort peak for the Locate peak consistence }
  if str_msg.length > 0 then INTEGR$REF_INTEGR := str_msg"address
                        else INTEGR$REF_INTEGR := nil
end INTEGR$REF_INTEGR;



end RPW_INT_COMPUTE.
