{
*************************************************************************
*                                                                       *
*                                                                       *
*        R P W D A T A  (Reduce Powder DATA for Diffraction)            *
*                                                                       *
*               ( ILL Data Base Manager Source File )                   *
*                                                                       *
*                 Version  1.1-B  - - 31-Oct-2009                       *
*                                                                       *
*                                by                                     *
*                                                                       *
*            Pierre Wolfers, Laboratoire de Cristallographie            *
*                                                                       *
*            CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166            *
*                                                                       *
*                       F 38042 GRENOBLE CEDEX 9                        *
*                                                                       *
*                             F R A N C E                               *
*                                                                       *
*                                                                       *
*                                                                       *
*************************************************************************

/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////

}

module RPWDATA_INP_OUT;


%include 'RPWSRC:rpwdata_env.pas' {, list_on};





procedure INIT_PATTERN( p: pat_ptr );
begin
  with p^, mai_parm, hkl_parm do
  begin
    next       :=          nil;                 { Default is to have no successor }
    idcnt      :=            0;                 { Set pattern index to null index value }
    numor      :=           -1;                 { No numor }
    dateh.length        :=   0;                 { Init the pattern identification to empty }
    sample.length       :=   0;
    comment.length      :=   0;
    monitor    :=            0;                 { No monito never counting time }
    cnttime    :=            0;

    dtheta     :=        th0_r;                 { Get initial 2 theta angle }
    omega      :=          0.0;                 { Set Eulrian Angle to 0.0 }
    chi        :=          0.0;
    phi        :=          0.0;
    tr1        :=          0.0;
    tr2        :=          0.0;
    step       :=        ths_r;                 { Set the fixed theta step }
    lambda1    :=        gw1_r;                 { No defined Wave length }
    lambda2    :=        gw2_r;
    rvp2       :=          0.0;                 { Init to zero all Free parameter fields }
    rvp3       :=          0.0;
    rvp4       :=          0.0;
    rvp5       :=          0.0;
    rvp6       :=          0.0;
    rvp7       :=          0.0;
    rvp8       :=          0.0;
    rvp9       :=          0.0;
    ivp1       :=            0;
    ivp2       :=            0;
    amin       :=        1e+10;
    amax       :=        -amin;
    min        :=        1e+10;
    max        :=        - min
  end
end INIT_PATTERN;



[global]
procedure GET_PATTERNS;
{ Procedure to read all the pattern in the input data file (D1B/D20 format) }
const
  mimast       = 1.0E+10;                       { Value to init the minimum and maximum }

var
  nda:               integer;




  procedure SKIP_INP_CHAR( nn: integer );
  var
    ch: char;

  begin
    for ii := 1 to nn do READ( inp, ch )
  end SKIP_INP_CHAR;



  procedure ELLIMINATE_CR( var st: string );
  const
    CR      = CHR( 13 );                        { ASCII Character 13 to elliminate }

  begin
    with st do
      if length > 0 then
        if body[length] = CR then length := length - 1
  end ELLIMINATE_CR;



  procedure GET_STD_FORMAT;
(* pat_rec( dim: integer ) = record { * Definition of Pattern record }
  next:                                pat_ptr;             { Pointer to next pattern }
  idcnt,                                                    { Pattern index }
  numor:                               integer;             { Integer Identifier NUMOR }
  dateh,                                                    { Pattern date and hour }
  sample,                                                   { sample on the measure }
  comment:                        string( 64 );             { Pattern comment of 64 character max. }
  monitor,                                                  { Monitor count }
  cnttime,                                                  { Counting time }
  dtheta, omega, chi, phi,                                  { Initial characteristic angles }
  tr1, tr2, step,                                           { sample position X, Y values, theta step }
  t_set, t_reg, t_sample,                                   { Pattern temperature: Setting, on regulator, on sample }
  lambda1, lambda2,                                         { Wave lengths (old rvp1 free parameter) }
  rvp2, rvp3, rvp4, rvp5,                                   { Free real parameters rvp1..rvp9 }
  rvp6, rvp7, rvp8, rvp9:                 real;
  ista, iend,                                               { Scan index limit number for internal use }
  ivp1, ivp2:                          integer;             { Free integer parameters ivp1 and ivp2 }
  amin, amax,                                               { Deduced Minimaxi on 2*theta }
  min,  max:                              real;             { Deduced Minimum and maximum of intensity }
  dat:             array[1..dim] of data_point              { The pattern dim points }
end;
*)
  var
    comment,                                                { The current pattern comment }
    sample,                                                 { The sample specification }
    dath,                                                   { Date and hour }
    line:                                string;            { The current line }
    word:                          string( 32 );            { The current word }
    w_len1, w_len2, w_lenr,                                 { Wave length(s) and ratio }
    moni, time,                                             { Monitor and time }
    temp_s, temp_r, temp_c:                real;            { Temperature on sample, regulation and temperature setting }
    len, ip:                            integer;            { Data size in pattern points }


  procedure GET_NEW_LINE;
  begin
    line.length := 0;
    while not EOF( inp ) and (line.length = 0) do
    begin
      READLN( inp, line );
      while (line.length > 0) and (line[line.length] <= ' ') do line.length := line.length - 1;
      if line.length > 0 then
      begin
        if line[1] = '#' then
          if line.length = 1 then line.length := 0
          else
          begin
            SET_CASE( line, true );
            pi := 2;
            READV( line:ip, word )
          end
    end
  end GET_NEW_LINE;



  begin { GET_STD_FORMAT }
    GET_NEW_LINE;
    if line.length > 9 then line.length := 9;
    SETCASE( line );
    if line <> '# RPWDATA

  end GET_STD_FORMAT;



  procedure GET_D1B_FORMAT;
  var
    p:                                 pat_ptr; { Pointer to the new pattern record }
    line1, line2, line3, line4:         string; { String to keep the first four lines }
    i, len, nlupat:                    integer;
    mi, ma:                               real;
    bs:                                boolean;
    sp:                                   char; { Space character to read }

  begin
    READLN( inp, nlupat );                      { Ignore the first line (where the number of pattern can be specified) }
    while not EOF( inp ) do                     { Loop on all pattern(s) }
    begin
      READLN( inp, line1 );
    exit if EOF( inp );
      READV( line1, line2:0:true );             { Try to find the "-10000" of the end of data file }
    exit if line2 = '-10000';
      READLN( inp, line2 );
      READLN( inp, line3 ); READLN( inp, line4 );
      READV( line4, len );                      { Get the size of pattern }

      NEW( p, len );                            { Create the pattern record }
      with p^ do
      begin
        READV( line1, dateh:20, sp, sample:10, sp, comment );   { Get the date, sample name, and comment }
        READV( line2, idcnt, numor );           { Get the pattern index and NUMOR }
        READV( line3, monitor, cnttime, dtheta, { Get the monitor and counting time, ... }
                      omega,   chi,     phi,    { ... Possible Eulerian angles, ... }
                      tr1,     tr2,     step,   { ... X,Y sample position and 2theta step, ... }
                      t_set,   t_reg,   t_sample );     { ... Temperatures setting, on regulation and on sample. }
        if dat_celsius or (t_sample <= 0.0) then
        begin                                   { Any null or negative temperature value is ... }
          t_set    := t_set    + Ice_Temp;      { ... taken as provided in Celsius degrees }
          t_reg    := t_reg    + Ice_Temp;
          t_sample := t_sample + Ice_Temp
        end;
        READV( line4, len,     lambda1, rvp2,   { Get the Size of pattern, wave Length and all free parameters }
                      rvp3,    rvp4,    rvp5,
                      rvp6,    rvp7,    rvp8,
                      rvp9,    ivp1,    ivp2 );
        lambda2 := 0.0;
        amin := dtheta;                         { Set the minimum and maximum of 2*theta }
        amax := dtheta + (dim - 1)*step;
        mi := mimast; ma := - mi;

        i := 0;
        ista := 1; iend := dim;                 { Set the default using }
        while i < dim do
        begin
          with dat[i + 1] do
          begin
            theta := dtheta + i*step;
            READ( inp, flag, int );             { Read each point of the diagram }
            sig := SQRT( ABS( int ) );
            mflg := [];                         { Set the marker flag to empty }
            if flag <= 0 then mflg := [mk_invalid];
            { Set the diagram minimum and maximum }
            if mi > int then mi := int
                        else if ma < int then ma := int
          end;
          i := i + 1
        end;
        { Set pattern minimaxi }
        min := mi; max := ma;
        { Update the general minimaxi }
        if tmin > t_set then tmin := t_set
                        else if tmax < t_set then tmax := t_set;
        if thmin > amin then thmin := amin;
        if thmax < amax then thmax := amax;
        if gmin > mi then gmin := mi;
        if gmax < ma then gmax := ma
      end;
      { Set the pattern in the pattern list }
      if pat_first = nil then pat_first := p
                         else pat_last^.next := p;
      pat_last := p;
      { Set this pattern in selection table if required }
      npat := npat + 1;
      bs := false;
      for ij := 1 to ndia do  if npat = tbnp[ij] then bs := true;
      for ij := 1 to ndiat do
        if fc_celsius in funcs then
        begin
          if ABS( p^.t_set - tbtem[ij] - Ice_Temp ) < 1.0E-3 then bs := true
        end
        else if ABS( p^.t_set - tbtem[ij] ) < 1.0E-3 then bs := true;
      if bs then
      begin { Form the mini-maxi of selected diagrams }
        if smin > mi then smin := mi;
        if smax < ma then smax := ma;
        { Put the selected diagram in the selection table }
        nda := nda + 1; tbpat[nda] := p
      end;
      READLN( inp );                            { Skip to end of current count line }
      READLN( inp, nlupat )                     { Read the End Pattern mark (should be -1000) }
    end
  end GET_D1B_FORMAT;



  procedure GET_XYD_FORMAT;
  const
    pat_msize = 1000000;                        { Maximum pattern size }

  var
    p:                                 pat_ptr; { Pointer to the new pattern record }
    nc:                                integer; { Pattern size }
    word:               [static]  string( 32 );
    line:               [static] string( 255 ); { Current line }
    xfc, yfc, stpin,                            { Internal X and Y multiplicators, and stepin }
    xmi, ymi, xma, yma, xx, yy, sg:       real;
    ch: char;
    ip, np, interp:                    integer;
    hd, bt: boolean;

  begin
    { Size the Pattern }
    nc := -6;                                   { Initialize the pattern size count (with six lines for the header) }
    while not EOF( inp ) do
    begin
      READLN( inp, line );
      if line.length > 0 then
        if (nc < 0) or 
           ((nc >= 0) and
           (line[1] <> '!') and (line[1] <> '#')) then
          nc := nc + 1                          { Do not count the comment lines in the pattern }
    end;
    CLOSE( inp );
    RESET( inp, inp_name );                     { Re-Open the data file to simulate a rewind }

    READLN( inp, line );                        { Read the first XYDATA Line }
    while (line.length > 0) and (line[line.length] < ' ') do line.length := line.length - 1;
    ip := 1;
    word.length := 0;
    while (ip < line.length) and (word.length = 0) do   { Look for the word "XYDATA" }
    begin
      READV( line:ip, word:16:true );
      if word = '#' then word.length := 0       { Ignore initial "#" character }
    end;
    SET_CASE( word, true );                     { Put the word in Capital characters }
    hd := word = 'XYDATA';                      { Select the header mode }
    NEW( p, nc );                               { Allocate the pattern record }
    with p^, mai_parm, hkl_parm do
    begin
      xfc := 1.0; yfc := 1.0;
      stpin := 0.0001; interp := 0;
      next       :=      nil;                   { Init the pattern link pointer }
      idcnt      :=        0;                   { The Pattern index, }
      numor      :=       -1;                   { No numor, }
      dateh.length   :=    0;                   { Default with no Comments }
      sample.length  :=    0;
      comment    :=  '*** No Comment ***';
      monitor := 0.0; cnttime := 0.0;           { Default monitor and time }
      dtheta  := 0.0; omega   := 0.0;
      chi := 0.0; phi := 0.0; tr1 := 0.0; tr2 := 0.0;
      step   := 0.001;
      t_set    := 0.0; t_reg    := 0.0; t_sample := 0.0;
      lambda1  := gw1_r; lambda2  := gw2_r; rwave2 := fwr_r;
      rvp2 := 0.0; rvp3 := 0.0; rvp4 := 0.0; rvp5 := 0.0;
      rvp6 := 0.0; rvp7 := 0.0; rvp8 := 0.0; rvp9 := 0.0;
      ista :=   0; iend :=   0; ivp1 :=   0; ivp2 :=   0;
      if hd then
      begin
        READV( line:ip, ch, comment );          { Set the comment }
        for ii := 1 to 5 do                     { Loop on all others header lines }
        begin
          READLN( inp, line );
          while (line.length > 0) and (line[line.length] < ' ') do line.length := line.length - 1;
          ip := 1;
          word.length := 0;
          while (ip < line.length) and (word.length = 0) do     { Look for the word "XYDATA" }
          begin
            READV( line:ip, word:16:true );             { Get a word }
            if word = '#' then word.length := 0         { Ignore initial "#" character }
          end;
          SET_CASE( word );                             { Put the word in Capital characters }
          if word = 'INTER' then
            READV( line:ip, xfc, yfc, interp, stpin )
          else
          if (word = 'TEMP') or (word = 'TEMPERATURE') then
          begin
            READV( line:ip, t_sample );
            if dat_celsius or (t_sample <= 0.0) then    { Any null or negative temperature value is ... }
              t_sample := t_sample + Ice_Temp;          { ... taken as provided in Celsius degrees }
            t_reg := t_sample; t_set := t_reg
          end
          else
          if SUBSTR( word, 1, 7 )  = 'MONITOR' then
          begin
            bt := false;
            while (ip < line.length) and (line[ip] <> ':') do ip := ip + 1;
            ip := ip + 1;
            word.length := 0;
            if ip < line.length then
              READV( line:ip, monitor, word:0:true );
            if word.length > 0 then READV( word, cnttime )
          end
          else
          if word = 'LAMBDA' then READV( line:ip, lambda1 )
          else
          if word = 'LAMBDA2' then READV( line:ip, lambda2 )
          else
          if word = 'LAMBDAR' then READV( line:ip, rwave2 )
          else
          if word = 'SAMPLE' then READV( line:ip, ch, sample )
          else
          if word = 'DATEH'  then READV( line:ip, ch, dateh )
          else
          if word = 'ORIENT' then READV( line:ip, ch, omega, chi, phi )
        end
      end
      else
      begin
        comment := line;
        for ii := 1 to 5 do READLN( inp )       { Ignore all not header lines }
      end;

      np := 1;
      while np <= nc do
      with dat[np] do
      begin
        READLN( inp, line );
        if line[line.length] < ' ' then line.length := line.length - 1;
        if line.length > 0 then
          if (line[1] <> '#') and (line[1] <> '!') then
          begin
            ip := 1;
            READV( line:ip, xx, yy, word:0:true );
            if word.length > 0 then READV( word, sg )
                               else sg := SQRT( ABS( yy ) );
            xx := xx*xfc; yy := yy*yfc; sg := yfc*sg;
            if np = 1 then { * For the first point }
            begin
              xmi := xx; xma := xx;
              ymi := yy; yma := yy;
              dtheta := xx*xfc
            end
            else
            begin
              if xmi > xx then xmi := xx
                          else if xma < xx then xma := xx;
              if ymi > yy then ymi := yy
                          else if yma < yy then yma := yy;
            end;
            flag := 1; theta := xx; int := yy; sig := sg;
            mflg := [];                         { Set the marker flag to empty }
            np := np + 1
          end;
      end;
      amin  :=      xmi; amax  :=      xma;
      min   :=      ymi; max   :=      yma;
      { Set the global mini maxi for this unique pattern }
      smin  :=      ymi; smax  :=      yma;
      tmin  := t_sample; tmax  := t_sample;
      thmin :=      xmi; thmax :=      xma;
      gmin  :=      ymi; gmax  :=      yma
    end;
    { Set the pattern with this unique pattern }
    npat := npat + 1;
    nda := 1; tbpat[1] := p;
    if pat_first = nil then pat_first := p
                       else pat_last^.next := p;
    pat_sel   := p;
    pat_last  := p
  end GET_XYD_FORMAT;



  procedure GET_INLRX_FORMAT;
  var
    p:                         pat_ptr;
    line:                       string;
    nc, i:                     integer;
    th, mi, ma:                   real;
    bs:                        boolean;
    ch:                           char;

  begin
    repeat                                      { Loop on all pattern(s) }
      nc := 0;                                  { Initialize the number of point in one pattern }
      { Look for "RAW" word }
      repeat
        READLN( inp, line );
        if EOF( inp ) then nc := -1;
      until (nc < 0) or ((line[1] = 'R') and (line[2] = 'A') and (line[3] = 'W'));
    exit if nc < 0;
      READLN( inp, nc );                        { Read the size (in point) of the pattern }
      NEW( p, nc );                             { Allocate  a pattern record for nc points }
      with p^, mai_parm, hkl_parm do
      begin                                     { Now we are filling the pattern record }
        next       :=          nil;             { Default is to have no successor }
        idcnt      :=            0;             { Set pattern index to null index value }
        numor      :=           -1;             { No numor }
        monitor    :=            0;             { No monitor }

        dtheta     :=        th0_r;             { Get initial 2 theta angle }
        omega      :=          0.0;             { Set Eulrian Angle to 0.0 }
        chi        :=          0.0;
        phi        :=          0.0;
        tr1        :=          0.0;
        tr2        :=          0.0;
        step       :=        ths_r;             { Set the fixed theta step }
        rvp2       :=          0.0;             { Init to zero all Free parameter fields }
        rvp3       :=          0.0;
        rvp4       :=          0.0;
        rvp5       :=          0.0;
        rvp6       :=          0.0;
        rvp7       :=          0.0;
        rvp8       :=          0.0;
        rvp9       :=          0.0;
        ivp1       :=            0;
        ivp2       :=            0;

        READLN( inp, cnttime );                 { Read the counting time }
        for ii := 1 to 7 do READLN( inp );      { Skip the next seven lines }
        READLN( inp, comment );                 { Read the Pattern comment }
        ELLIMINATE_CR( comment );
        READLN( inp, lambda1 );                 { Read the Wave length for alpha1 and alpha 2 }
        READLN( inp, lambda2 );
        rwave2 := fwr_r;                        { force the default Wave length ratio }
        { if gw1_r > 0.0 then lambda1 := gw1_r; { Force the Lambda when it is externaly specified }
        { if gw2_r > 0.0 then lambda2 := gw2_r; { Force the Lambda when it is externaly specified }
(*
        READLN( inp, dateh );                   { Get the date and hour of Measurement >>> Actually ignored }
        SKIP_INP_CHAR( 12 );
*)
        READLN( inp, ch, dateh );
        ELLIMINATE_CR( dateh );
        numor := npat + 1;                      { Set the integer identifier }
        SKIP_INP_CHAR( 12 );
        READLN( inp, sample );                  { Get sample field }
        ELLIMINATE_CR( sample );
        SKIP_INP_CHAR( 15 );                    { Skip 15 characters }
        READLN( inp, t_sample );                { Get the temperature }
        t_sample := t_sample + Ice_Temp;        { Convert it in Kelvin (always in Celsius) }
        t_set    := t_sample;                   { Default assignements }
        t_reg    := t_sample;
        READLN( inp );                          { Skip one line }
        READLN( inp );                          { Skip one line }
        repeat
          READLN( inp, line );
        until line[1] = '0';                    { Look for the begin of pattern Counts }
        i := 0;
        amin := dtheta;                         { Set the minimum and maximum of 2*theta }
        amax := dtheta + nc*step;
        mi := mimast; ma := - mi;
        ista := 1; iend := nc;
        while i < nc do
        begin
          with dat[i + 1] do
          begin
            flag := 1;                          { Set as a valid point }
            theta := dtheta + i*step;
            READLN( inp, int );                 { Read each point of the diagram }
            sig := SQRT( ABS( int ) );          { Set the sigma field }
            mflg := [];                         { Set the marker flag to empty }
            { Set the diagram minimum and maximum }
            if mi > int then mi := int
                        else if ma < int then ma := int
          end;
          i := i + 1
        end;
        { Set pattern minimaxi }
        min := mi; max := ma;
        { Update the general minimaxi }
        if tmin > t_set then tmin := t_set
                        else if tmax < t_set then tmax := t_set;
        if thmin > amin then thmin := amin;
        if thmax < amax then thmax := amax;
        if gmin > mi then gmin := mi;
        if gmax < ma then gmax := ma
      end;
      { Set the pattern in the pattern list }
      if pat_first = nil then pat_first := p
                         else pat_last^.next := p;
      pat_last := p;
      { Set this pattern in selection table if required }
      npat := npat + 1;
      bs := false;
      for ij := 1 to ndia do  if npat = tbnp[ij] then bs := true;
      for ij := 1 to ndiat do
        if fc_celsius in funcs then
        begin
          if ABS( p^.t_set - tbtem[ij] - Ice_Temp ) < 1.0E-3 then bs := true
        end
        else if ABS( p^.t_set - tbtem[ij] ) < 1.0E-3 then bs := true;
      if bs then
      begin { Form the mini-maxi of selected diagrams }
        if smin > mi then smin := mi;
        if smax < ma then smax := ma;
        { Put the selected diagram in the selection table }
        nda := nda + 1; tbpat[nda] := p
      end
    until EOF( inp )
  end GET_INLRX_FORMAT;



  procedure GET_SRAW_FORMAT;
  var
    len, pid:                  integer;
    mi, ma, th0, stp, thnd:     double;
    p:                         pat_ptr;
    bs:                        boolean;

  begin
    pid := 0;                                   { Init the pattern index }
    while not EOF( inp ) do                     { Loop  for each pattern to get }
    begin
      READ( inp, th0 );                         { Get the start, step and end angles (2*theta) }
    exit if EOF( inp );
      READ( inp, stp );
    exit if EOF( inp );
      READLN( inp, thnd );
    exit if EOF( inp );
      pid := pid + 1;
      len := ROUND( (thnd - th0)/stp ) + 1;     { Compute the size of pattern }
      thnd := (len + 0.5)*stp + th0;            { Adjust the end angle }
      NEW( p, len );                            { Allocate a new pattern }
      INIT_PATTERN( p );                        { Initialize all pattern fields }
      with p^, mai_parm, hkl_parm do
      begin
        idcnt  :=  pid;                         { Set the local identifier }
        if ABS( th0 ) < 180.0 then              { Set specific 2*theta origine and step ...when legal value }
        begin  dtheta :=  th0; step   :=  stp  end;     { ... when the th0 value is legal }
        amin := dtheta;                         { Set the 2*theta mini-maxi }
        amax := dtheta + (len - 1)*step;
        mi := min; ma := max;                   { Get the initial minmaxi (from INIT_PATTERN) }
        lambda1  := gw1_r; lambda2  := gw2_r;   { Set the default Wave-Length(s) }
        rwave2 := fwr_r;                        { Set Wave Lengths ratio }
        for ii := 1 to len do
        with dat[ii] do
        begin
          theta := (ii - 1)*stp + th0;          { Set the 2*theta value }
          int   := 0.0;                         { Init and read the intensity }
          if not EOF( inp ) then READ( inp, int );
          sig := SQRT( ABS( int ) );            { Set a default sigma }
          mflg := [];                           { Set the marker flag to empty }
          if mi > int then mi := int;           { set the intensity Mini-Maxi }
          if ma < int then ma := int
        end;
        { Set pattern minimaxi }
        min := mi; max := ma;
        { Update the general minimaxi }
        if tmin > t_set then tmin := t_set
                        else if tmax < t_set then tmax := t_set;
        if thmin > amin then thmin := amin;
        if thmax < amax then thmax := amax;
        if gmin > mi then gmin := mi;
        if gmax < ma then gmax := ma
      end;
      { Set the pattern in the pattern list }
      if pat_first = nil then pat_first := p
                         else pat_last^.next := p;
      pat_last := p;
      { Set this pattern in selection table if required }
      npat := npat + 1;
      bs := false;
      for ij := 1 to ndia do  if npat = tbnp[ij] then bs := true;
      for ij := 1 to ndiat do
        if fc_celsius in funcs then
        begin
          if ABS( p^.t_set - tbtem[ij] - Ice_Temp ) < 1.0E-3 then bs := true
        end
        else if ABS( p^.t_set - tbtem[ij] ) < 1.0E-3 then bs := true;
      if bs then
      begin { Form the mini-maxi of selected diagrams }
        if smin > mi then smin := mi;
        if smax < ma then smax := ma;
        { Put the selected diagram in the selection table }
        nda := nda + 1; tbpat[nda] := p
      end;
    end
  end GET_SRAW_FORMAT;



  procedure GET_COL_FORMAT;
  const
    pat_msize = 1000000;                        { Maximum pattern size }

  var
    p:                                 pat_ptr; { Pointer to the new pattern record }
    nc:                                integer; { Pattern size }
    word:               [static]  string( 32 );
    line:               [static] string( 255 ); { Current line }
    xfc, yfc, stpin,                            { Internal X and Y multiplicators, and stepin }
    xmi, ymi, xma, yma, xx, yy, sg:       real;
    ch: char;
    ip, np, interp:                    integer;
    hd, bt: boolean;

  begin
    while not EOF( inp ) do
    begin
      READLN( inp, line );                      { Get a line of the file }

    end;

  end GET_COL_FORMAT;



begin { GET_PATTERNS }
  { Initialize the Selected pattern array table }
  for i := 1 to npat_max do  tbpat[i] := nil;
  nda  := 0;                                    { No selection }
  npat := 0;                                    { No pattern in memory }

  OPEN( inp, inp_name, [read_file,error_file] );{ Open the data file }
  if iostatus <> 0 then
  begin
    WRITELN( ' *** RPW Error: Cannot open the file "', inp_name, '". ***' );
    PASCAL_EXIT( 4 )
  end;

  tmin  := mimast; tmax  := - tmin;             { Initialize all minimaxi }
  thmin := mimast; thmax := -thmin;
  gmin  := mimast; gmax  := - gmin;
  smin  := mimast; smax  := - smin;

  case inp_frm of
(*
    pf_standard:   GET_STD_FORMAT;
*)

    pf_standard:   GET_STD_FORMAT;

    pf_d1b:        GET_D1B_FORMAT;

    pf_xydata:     GET_XYD_FORMAT;

    pf_macinelrx:  GET_INLRX_FORMAT;

    pf_singleraw:  GET_SRAW_FORMAT;

  { pf_column:     GET_COL_FORMAT; }

  otherwise
    WRITELN( ' *** RPW Error: Unsupported or unimplemented input format. ***' );
    PASCAL_EXIT( 2 )
  end;

  CLOSE( inp );                                 { Close the data file }

  WRITELN( ' RPW Found ', npat:0, ' patterns in the file "', inp_name, '".' );
  if npat = 0 then
  begin
    WRITELN( ' *** RPW Error: Cannot work without Pattern and exit.' );
    PASCAL_EXIT( 4 )
  end
  else
  begin
    WRITELN( ' ':24, 'Global Statistics' );
    WRITELN( ' ':6, 'Minimum and maximum of 2*Theta Angle: ', thmin:8:3, ', ', thmax:8:3, ',' );
    WRITELN( ' ':6, 'Minimum and maximum of Temperature:   ', tmin:8:2,  ', ', tmax:8:2,  ',' );
    WRITELN( ' ':6, 'Minimum and maximum of intensities:   ', gmin:8:3,  ', ', gmax:8:3,  '.' );
    if npat = 1 then
    begin
      ndia := 1;
      tbpat[1] := pat_first;
      if fc_3dplot in funcs then funcs := funcs + [fc_plot,fc_ndia]
    end
    else
      if ndia = 0 then
        if fc_ndia in funcs then ndia :=  nda
                            else ndia := npat;
  end;
end GET_PATTERNS;



[global]
procedure WRITE_NORMALIZED_FILE;
var
  p:                           pat_ptr;
  ij:                          integer;
  sn:                  [static] string;
  bsingle, bemp:               boolean;

(*
    pf_standard,                                { Future standard format }
    pf_d1b,                                     { D1b/D20 file format }
    pf_column,                                  { Column format }
    pf_xydata,                                  { XY-DATA format }
    pf_macinelrx,                               { X-Ray Inel on Macintosh format }
    pf_null                                     { Null format for unspecified format }
*)

  procedure OUT_PATTERN( p: pat_ptr; nb: integer );
  var
    flin: integer;
    step:    real;

  begin
    with p^ do
      case out_frm of
        pf_standard:        { The standard future default file format in the future }
          begin
            WRITELN( out, '# PATTERN ', comment );
            WRITELN( out, '# LAMBDA ', lambda1:10:6, ' LAMBDA2 ', lambda2:10:6, ' LAMBDAR ', rwave2:10:6 );
            WRITELN( out, '# TEMP ', t_sample:8:2, ' ', t_reg:8:3, ' ', t_set:8:3 );
            WRITELN( out, '# Monitor/counts, Counting time/sec:{ Monitor }', monitor:12:0, ' ', cnttime:9:0 );
            WRITELN( out, '# DATEH', dateh, ' SAMPLE ', sample );
            WRITELN( out, '# ORIENT ', { Omega } omega:8:3, ' ', { Chi } chi:8:3, ' ', { Phi } phi:8:3, ' ' );
            WRITELN( out, '# DATA ', dim:8 );
            for idx := 1 to dim do
              with dat[idx] do
                if not (mk_invalid in mflg) then WRITELN( out, theta:8:3, ' ', int:10:0, sig:6:0 );
            WRITELN( out, '# EOD' )
          end;


        pf_column: { Column Mode }
          for ij := 1 to dim do
            with dat[ij] do
            begin
              if mk_invalid in mflg then flin := 0
                                    else flin := flag;
              WRITELN( out, ij:6, ' ', flin, theta:10:3, int:10:0, sig:10:0 )
            end;

        pf_d1b: { D1B/D20 Mode }
          begin
            step := ROUND( (amax - amin)/(dim - 1) );
            WRITELN( out, dateh:20, sample:10, ' ', comment );
            WRITELN( out, nb:3, ' ', numor:6, ' ', lambda1:8:4, ' ', lambda2:8:4, ' ', rwave2:8:5 );
            WRITELN( out, { Monitor } monitor:12:0, ' ', { Time }  cnttime:9:0, ' ',
                          { 2theta0 }   dtheta:8:3, ' ', { Omega }   omega:8:3, ' ',
                          { Chi }          chi:8:3, ' ', { Phi }       phi:8:3, ' ',
                                           tr1:8:3, ' ',               tr2:8:3, ' ', { Step }      step:8:3, ' ',
                                         t_set:8:3, ' ',             t_reg:8:3, ' ',           t_sample:8:3 );
            WRITELN( out, dim:4, ' ',  lambda1:9:3, ' ', rvp2:9:3, ' ', rvp3:9:3, ' ',
                                          rvp4:9:3, ' ', rvp5:9:3, ' ', rvp6:9:3, ' ',
                                          rvp7:9:3, ' ', rvp8:9:3, ' ', rvp9:9:3, ' ',
                                          ivp1:6,   '   ', ivp2:6 );
            for idx := 1 to dim do
              with dat[idx] do
              begin
                if mk_invalid in mflg then WRITE( out, ' 0' )
                                      else WRITE( out, flag:2 );
                WRITE( out, int:8:0 );
                if idx mod 10 = 0 then WRITELN( out )
              end;
            if dim mod 10 <> 0 then WRITELN( out );
            WRITELN( out, -1000:10 )
          end;

        pf_xydata: { XY_DATA Mode }
          begin
            WRITELN( out, '# XYDATA ', comment );
            WRITELN( out, '# INTER  1.0  1.0  0  0.01000   LAMBDA ', lambda1:10:6, ' LAMBDA2 ', lambda2:10:6, ' LAMBDAR ', rwave2:10:6 );
            WRITELN( out, '# TEMP ', t_sample:8:2, ' ', t_reg:8:3, ' ', t_set:8:3 );
            WRITELN( out, '# Monitor/counts, Counting time/sec:{ Monitor }', monitor:12:0, ' ', cnttime:9:0 );
            WRITELN( out, '# DATEH', dateh, ' SAMPLE ', sample );
            WRITELN( out, '# ORIENT ', { Omega } omega:8:3, ' ', { Chi } chi:8:3, ' ', { Phi } phi:8:3, ' ' );
            for idx := 1 to dim do
              with dat[idx] do
                if not (mk_invalid in mflg) then WRITELN( out, theta:8:3, ' ', int:10:0, sig:6:0 )
          end;

        pf_singleraw: { Single raw mode }
          begin
          end;

      otherwise
        WRITELN( ' *** RPW Command error : The specified output file format is not currently implemented. ***' );
        PASCAL_EXIT( 2 )
      end
  end OUT_PATTERN;



  function OUT_HEADER( in_var fname: string; nd: integer ): boolean;
  var
    br: boolean := false;

  begin
    REWRITE( out, fname );
    case out_frm of
      pf_standard:
        begin
          WRITELN( out, '# RPWDATA' )
        end;

      pf_d1b:
        begin
          WRITELN( out, nd:5 )
        end;

    otherwise
      br := true                                { All other format can be include only one pattern }
    end;
    OUT_HEADER := br
  end OUT_HEADER;



  procedure OUT_TRAILER;
  begin
    case out_frm of
      pf_standard:
        begin
          WRITELN( out, '# EOF' )
        end;

      pf_d1b:
        begin
          WRITELN( out, -10000:10 )
        end;

    otherwise
    end;
    CLOSE( out )
  end OUT_TRAILER;



begin { WRITE_NORMALIZED_FILE }
  if ndia > 0 then                              { One or more Pattern are selected in command line }
    if ndia > 1 then
    begin                                       { Many Pattern to output }
      bsingle := OUT_HEADER( out_name, ndia );  { bn flags when the format is only for one pattern }
      p := pat_first;
      ij := 0;
      repeat
        ij := ij + 1;
        with p^ do
        begin
          if bsingle then
          begin
            WRITEV( sn, out_name, '_', ij:-4 );
            bemp := OUT_HEADER( sn, ij )
          end;
          OUT_PATTERN( tbpat[ij], ij );
          if bsingle then OUT_TRAILER;
          p := next
        end
      until ij >= ndia;
      if not bsingle then OUT_TRAILER
    end
    else
    begin
      bemp := OUT_HEADER( out_name, 1 );
      OUT_PATTERN( tbpat[1], 1 );               { Only One Pattern to Output }
      OUT_TRAILER
    end
  else                                          { No selected pattern in the command line }
    if npat > 1 then
    begin                                       { Many Pattern to output by default }
      bsingle := OUT_HEADER( out_name, ndia );  { bn flags when the format is only for one pattern }
      p := pat_first;
      ij := 0;
      repeat
        ij := ij + 1;
        with p^ do
        begin
          if bsingle then
          begin
            WRITEV( sn, out_name, '_', ij:-4 );
            bemp := OUT_HEADER( sn, ij )
          end;
          OUT_PATTERN( p, ij );
          if bsingle then OUT_TRAILER;
          p := next
        end
      until p = nil;
      if not bsingle then OUT_TRAILER
    end
    else                                        { One Pattern to Output }
    begin
      bemp := OUT_HEADER( out_name, 1 );
      OUT_PATTERN( pat_first, 1 );              { Only one Pattern to output }
      OUT_TRAILER
    end
end WRITE_NORMALIZED_FILE;



end RPWDATA_INP_OUT.
