{
*************************************************************************
*                                                                       *
*                                                                       *
*        R P W D A T A  (Reduce Powder DATA for Diffraction)            *
*                                                                       *
*                2D DRAW Interface Graphic Module                       *
*                                                                       *
*             ( ILL Data Base Manager Source File )                     *
*                                                                       *
*                Version  1.1-C  - - 30-Nov-2009                        *
*                                                                       *
*                                by                                     *
*                                                                       *
*                  Pierre Wolfers, Institut Neel                        *
*                                                                       *
*          CNRS GRENOBLE,  25 Avenue des Martyrs, B.P. 166              *
*                                                                       *
*                     F 38042 GRENOBLE CEDEX 9                          *
*                                                                       *
*                           F R A N C E                                 *
*                                                                       *
*                                                                       *
*                                                                       *
*************************************************************************

/////////////////////////////////////////////////////////////////////////
//                                                                     //
//                                                                     //
//                  Global Public Licence (GPL)                        //
//                                                                     //
//                                                                     //
// This license described in this file overrides all other licenses    //
// that might be specified in other files for this library.            //
//                                                                     //
// This library is free software; you can redistribute it and/or       //
// modify it under the terms of the GNU Lesser General Public          //
// License as published by the Free Software Foundation; either        //
// version 2.1 of the License, or (at your option) any later version.  //
//                                                                     //
// This library is distributed in the hope that it will be useful,     //
// but WITHOUT ANY WARRANTY; without even the implied warranty of      //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU   //
// Library General Public License for more details.                    //
//                                                                     //
// You should have received a copy of the GNU Lesser General Public    //
// License along with this library (see COPYING.LIB); if not, write to //
// the Free Software Foundation :                                      //
//                      Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  //
//                                                                     //
/////////////////////////////////////////////////////////////////////////

}
program RPW_DATA;

%include 'RPWSRC:rpwdata_env' {, list_on};




  {**********************************************************************************
  *                                                                                 *
  *                       Global  Variables  Declarations                           *
  *                                                                                 *
  **********************************************************************************}

const

  Const_def_setup   = 'rpw_setup.rpw_data';                     { Default name for setup file }

  { Default Setup Label Block }
  mai_prmdef = mai_prmty[    '*** RWDATA Default Setup ***',    { tit_s : Job Title }
                             'Created the 11-oct-2007',         { lb1_s : First comment line }
                             '',     '',    '',                 { lb%_s : (2..4) Comment line # 2,3,4 }
                             0.0,   0.02,                       { th0-r, ths_r : (2)Theta zero, and step }
                           1.1e3,  1.1e3,                       { tmi_r, tma_r : (2)Theta range (defaulted to no limits) }
                           false                                { tmd_f : Default is 2-theta coordinate }
                        ];

  drw_prmdef = drw_prmty[   -3.5,   -0.8,                       { hls_r, vls_r : Horizontal and Vertical Label Shift }
                             0.0,    0.0,    0.0,               { xsz_r, ysz_r, zsz_r : Sizes of Each Axes for 2d and 3D Plots }
                             0.2,                               { pcs_r : Count shift at 20% }
                            90.0,   80.0, -100.0,               { ome_r, chi_r, phi_r : Default 3D plot orientation (ome, chi, phi) }
                             0.0                                { dis_r: Default 3D View distance = 0 => Infinite perspective }
                        ];


[global]
var
  default_setup,                                                { File Setup Default }
  setupf:             linetype := 'rpw_setup.rpw_data';         { Setup file Specification string }

  str_msg,                                                      { String to build an output message }

  int_name,                                                     { Integration result file specification }
  out_name,                                                     { Output file specification }
  inp_name:                                   linetype;         { Input file specification }

  pat_sel,                                                      { Pointer to the selected pattern }
  pat_first,                                                    { Pointer to first and last pattern }
  pat_last:                    pat_ptr  :=         nil;

  crv_frs,                                                      { Pointer to first and last additional curev to display }
  crv_lst:                     crv_ptr  :=         nil;

  inp_frm,                                                      { input file format }
  out_frm:                  file_format := pf_standard;         { Output file format }

  inpl_frm,                                                     { Line format string for input }
  outl_frm:                    linetype     :=      '';         { Line format string for output }

  funcs:                                     functsety;         { Set of function to execute }

  ndiat,                                                        { Number of pattern to select by temperature }
  ndia,                                                         { Number of Pattern to Select by number }
  npat:                        integer  :=           0;         { Total number of pattern in memory }

  tbtem:                                   pattab1_flt;         { Table of Pattern temperature to select }
  tbnp:                                    pattab1_int;         { Table of Pattern number to select }

  tbpat:                                   pattab2_ptr;         { Table of Pattern pointer to select }
  tbknd:                                   pattab2_boo;         { Table of Line/Marker mode flags }
  tbcol,                                                        { Table of Line/Marker colors }
  tbtyp:                                   pattab2_int;         { Table of Line/Marker kinds }
  tbsiz:                                   pattab2_flt;         { Table of Line/Marker sizes }

  dstatus,                                                      { Return status for DRAW Graphic Library }
  irep:                                           Dint;         { Return value for DRAW$DIALOG Procedure }

  cshift,                                                       { Shift to apply at curve related to current pattern }
  e_thmin, e_thmax,                                             { Effective external current theta limits }
  smin,    smax,                                                { Minimum and Maximum of Selected Pattern }
  tmin,    tmax,                                                { Minimum and Maximum of temperature }
  thmin,   thmax,                                               { Minimum and Maximum of angle 2*theta }
  spthmin, spthmax,                                             { 2*theta mini-maxi of selected pattern }
  szthmin, szthmax,                                             { 2*theta mini-maxi of selected zone in the selected pattern }
  gmin,    gmax,                                                { General minimum and maximum }
  xs,      ys:                  Dfloat  :=         0.0;         { Paper size for draw }

  sel_ileft, sel_iright:        Dint    :=           0;         { Left and Right limits of Angular domain (Zone). }

  sel_pat:                      pat_ptr :=         nil;         { Current Selected Pattern pointer }

  hkl_free,                                                     { Free Hkl list }
  hkl_first,                                                    { HKL list header }
  hkl_last:                     hkl_ptr :=         nil;

  nerr_flg,                                                     { Flag for computing error during a fit }
  dat_celsius,                                                  { Flag for default Celcius Temperature in input }
  fil_nbr,                                                      { Flag for append pattern number to file name }
  out_app,                                                      { Flag for Output Append mode }
  drw_ok:                       Dbool   :=       false;         { Flag the graphic mode as enable (true) / Disable (false) }

  nzone,                                                        { Total number of defined integration zones }
  ntotpck,                                                      { Total number of defined diffraction peaks }
  nenapck,                                                      { Current number of Enabled peaks }

  frspint,                                                      { First pattern to integrate }
  nbrpint,                                                      { Number of pattern to integrate }
  ndat,                                                         { Number of data }
  np,                                                           { Total number of parameters }
  max_diag,                                                     { Maximum of allowed singularities (by LSQ cycle) }
  nsing:                       integer :=            0;         { Number of detected singularity }

  chi2eps,                                                      { Increasing tolerance for chi2 in Marquardt mode }
  chi2,                                                         { Chi_squared = Goodness of fit }
  lchi2,                                                        { Previous Chi_squared }
  nchi2:                                       lsq_flt;         { Chi_squared*(n-v) = Goodness of fit numerator }

  out,                                                          { Output for results }
  inp:                                            text;         { Input File }

  mai_parm:                                  mai_prmty;         { Main Setup }
  drw_parm:                                  drw_prmty;         { Graphic Setup }

  setup_tab:                              setup_tbtype;         { Environment table for setup }


var
  spc_stf,                                                      { Flag to signal the use of -setup option }
  mod_tty:                      Dbool   :=       false;         { Flag for Terminal mode (without graphic) }





  {**********************************************************************************
  *                                                                                 *
  *                    Procedure to initialize Main setup                           *
  *                                                                                 *
  **********************************************************************************}

  [global]
  procedure MAIN$SETUP_INIT;
  [static]
  var
    mai_desc: pdsc_tab( 12 ) := [ 0, 'Main Rpw Data Setup', 200,  95, [
                [ 'Identification of Job',                4,  43, 192,  48, prm_frm,  20, 3, 2, 0               ],

                [ 'Tit_s : Title of Setup',              30,  79, 162,   5, prm_str,  nil                       ],
                [ 'Ln1_s : Label line #1',               30,  71, 162,   5, prm_str,  nil                       ],
                [ 'Ln2_s : Label line #2',               30,  63, 162,   5, prm_str,  nil                       ],
                [ 'Ln3_s : Label line #3',               30,  55, 162,   5, prm_str,  nil                       ],
                [ 'Ln4_s : Label line #4',               30,  47, 162,   5, prm_str,  nil                       ],

                [ 'Default Global Parameters',            4,  12, 192,  26, prm_frm,  20, 3, 2, 0               ],

                [ 'tmd_f : Theta mode',                  44,  24,   5,   5, prm_bool, nil                       ],
                [ 'th0_r : (2)Theta Zero (Default pattern Start Angle)', 174,  24,  18,   5, prm_flt,  nil, -90.0, 90.0, 0.001 ],
                [ 'tmi_r : (2)Theta range from',         44,  16,  20,   5, prm_flt,  nil,   0.0, 180.0,  0.001 ],
                [ 'tma_r : to ',                         71,  16,  20,   5, prm_flt,  nil,   0.0, 180.0,  0.001 ],
                [ 'ths_r : (2)Theta Step (Default)',    174,  16,  18,   5, prm_flt,  nil,   0.0,   0.5,  0.001 ]
              ] ];

    drw_desc: pdsc_tab( 12 ) := [ 0, 'Graphic Rpw Data Setup', 140,  90, [
                [ 'Axis System Data',                    75,  12,  60,  56, prm_frm,  20, 3, 2, 0                ],

                [ 'hls_r : Label Shift (cm)  X',        115,  50,  15,   5, prm_flt,  nil,    0.0,  10.0,   0.01 ],
                [ 'vls_r : Y',                          115,  42,  15,   5, prm_flt,  nil,    0.0,  10.0,   0.01 ],
                [ 'xsz_r : Axes Sizes (cm)  X',         115,  32,  15,   5, prm_flt,  nil,    5.0, 300.0,   0.01 ],
                [ 'ysz_r : Y',                          115,  24,  15,   5, prm_flt,  nil,    5.0, 300.0,   0.01 ],
                [ 'zsz_r : Z',                          115,  16,  15,   5, prm_flt,  nil,    5.0, 300.0,   0.01 ],
                [ 'pcs_r : Successive Pattern Count Shift (rate)', 80,   4,  15,   5, prm_flt,  nil,    0.0,   1.0,   0.01 ],

                [ '3D View (Euler Angles)',               4,  12,  60,  56, prm_frm,  20, 3, 2, 0                ],

                [ 'ome_r : Omega',                       40,  50,  20,   5, prm_flt,  nil, -180.0, 180.0,   0.01 ],
                [ 'chi_r : Chi',                         40,  40,  20,   5, prm_flt,  nil,    0.0, 180.0,   0.01 ],
                [ 'phi_r : Phi',                         40,  30,  20,   5, prm_flt,  nil, -180.0, 180.0,   0.01 ],
                [ 'dis_r : 3D View Distance',            40,  16,  20,   5, prm_flt,  nil,    0.0, 500.0,   0.1  ]
              ] ];

    binit: boolean := true;

  begin
    if binit then
    begin
      with mai_desc, mai_parm do
      begin
        SET_REF( tb[ 2].rf, tit_s );
        SET_REF( tb[ 3].rf, lb1_s );
        SET_REF( tb[ 4].rf, lb2_s );
        SET_REF( tb[ 5].rf, lb3_s );
        SET_REF( tb[ 6].rf, lb4_s );
        SET_REF( tb[ 8].rf, tmd_f );
        SET_REF( tb[ 9].rf, th0_r );
        SET_REF( tb[10].rf, tmi_r );
        SET_REF( tb[11].rf, tma_r );
        SET_REF( tb[12].rf, ths_r )
      end;
      with drw_desc, drw_parm do
      begin
        SET_REF( tb[ 2].rf, hls_r );
        SET_REF( tb[ 3].rf, vls_r );
        SET_REF( tb[ 4].rf, xsz_r );
        SET_REF( tb[ 5].rf, ysz_r );
        SET_REF( tb[ 6].rf, zsz_r );
        SET_REF( tb[ 7].rf, pcs_r );
        SET_REF( tb[ 9].rf, ome_r );
        SET_REF( tb[10].rf, chi_r );
        SET_REF( tb[11].rf, phi_r );
        SET_REF( tb[12].rf, dis_r )
      end;
      for ii := 0 to Max_Setup do  setup_tab[ii] := nil;
      setup_tab[MAI_IDX] := mai_desc"address;
      setup_tab[DRW_IDX] := drw_desc"address;
      binit := false
    end;
    mai_parm := mai_prmdef;
    drw_parm := drw_prmdef;
    BCK$DEFAULT_SETUP;
    REF$DEFAULT_SETUP;
    INDEX$DEFAULT_SETUP;
    PRO$DEFAULT_SETUP
  end MAIN$SETUP_INIT;



  procedure EXTRACT_SETUP_OPTION;
  var
    io, id:    integer;
    so:   string( 36 );
    sn:         string;
    fn:        boolean;

  begin
    io := 1;
    fn := false;
    while io < argc do
    begin
      so := argv[io]^;
      id := INDEX( so, '=' );
      if id <> 0 then so.length := id - 1;
      SET_CASE( so, false );
      if so = '-setup' then fn := true;
    exit if fn;
      io := io + 1
    end;
    if fn then
    begin
      spc_stf := true;
      so := SUBSTR( argv[io]^, id + 1 );
      if so.length > 0 then
      begin
        { Take of the "-setup=..." option of the argument list }
        io := io + 1;
        while io < argc do
        begin  argv[io-1] := argv[io]; io := io + 1  end;
        argc := argc - 1;
        WRITEV( setupf, 'rpw_', so, '_setup.rpw_data' );
      end
    end
    else setupf := Const_def_setup
  end EXTRACT_SETUP_OPTION;




  {**********************************************************************************
  *                                                                                 *
  *         Procedure and Function to manage the Options of Command line            *
  *                                                                                 *
  **********************************************************************************}


  procedure GET_PARAMETERS;
  { Procedure to get the input and output file specifications from the command arguments. }

  const
    opttab_sz = 37;
    opttab_sm = 14;
    opt_nval  = 64;
    frmtab_sz = 11;

  type
    opt_identy = ( opt_inpfrm,                                  { Input format option }
                   opt_outfrm,                                  { Output format option }
                   opt_outpat,                                  { Output pattern file specification }
                   opt_taxis,                                   { Select Celcius/Kelvin for the Temperature Axis }
                   opt_tempsel,                                 { For temperature pattern selection }
                   opt_label,                                   { Plot Label required }
                   opt_axisz,                                   { 2D/3D axis size request }
                   opt_shift,                                   { 2D Pattern shift specification }
                   opt_npatt,                                   { Specify the Number of pattern to get }
                   opt_pcolor,                                  { Pattern Color Selection option }
                   opt_pltlin,                                  { Plot Pattern Line attributes option }
                   opt_pltmrk,                                  { Plot Pattern Marker attributes option }
                   opt_dir,                                     { Create a directory list }
                   opt_lambda,                                  { Specify a wave length }
                   opt_thlim,                                   { Specify Theta range }
                   opt_deftemp,                                 { Specify Celcius unit input temperature }
                   opt_verbose,                                 { Verbose flag }
                   opt_term,                                    { Mode without graphic for batch computing }
                   opt_help,                                    { Help display }
                   opt_not_opt                                  { Not an option == ERROR }
                 );

    opt_entry = record                                          { Define an option entry }
                  opt_iden: opt_identy;
                  opt_name: string( opttab_sm )
                end;

    opt_tabty = array[1..opttab_sz] of opt_entry;               { Define the option table type }

    frm_entry = record                                          { Define a file format entry }
                  frm_iden: file_format;
                  frm_name: string( opttab_sm )
                end;

    frm_names = array[1..frmtab_sz] of frm_entry;               { Define the file format table type }


  const
    opt_table = opt_tabty[ [opt_inpfrm,  '-if'     ], [opt_inpfrm,  '-inp_format'     ],
                           [opt_outfrm,  '-of'     ], [opt_outfrm,  '-out_format'     ],
                           [opt_outpat,  '-o'      ], [opt_outpat,  '-output'         ],
                           [opt_taxis,   '-tu'     ], [opt_taxis,   '-t_unit_axis'    ],
                           [opt_label,   '-lab'    ], [opt_label,   '-label'          ],
                           [opt_npatt,   '-ns'     ], [opt_npatt,   '-pn_select'      ],
                           [opt_tempsel, '-ts'     ], [opt_tempsel, '-pt_select'      ],
                           [opt_shift,   '-psh'    ], [opt_shift,   '-pattern_shift'  ],
                           [opt_pcolor,  '-pc'     ], [opt_pcolor,  '-pattern_color'  ],
                           [opt_pltlin,  '-lk'     ], [opt_pltlin,  '-line_kind'      ],
                           [opt_pltmrk,  '-mk'     ], [opt_pltmrk,  '-marker_kind'    ],
                           [opt_dir,     '-d'      ], [opt_dir,     '-directory'      ],
                           [opt_lambda,  '-w'      ], [opt_lambda,  '-wave_length'    ],
                           [opt_thlim,   '-th'     ], [opt_thlim,   '-theta_range'    ],
                           [opt_axisz,   '-ax'     ], [opt_axisz,   '-axis_sizes'     ],
                           [opt_deftemp, '-celcius'],
                           [opt_term,    '-tt'     ], [opt_term,    '-terminal'       ],
                           [opt_verbose, '-v'      ], [opt_verbose, '-verbose'        ],
                           [opt_help,    '-h'      ], [opt_help,    '-help'           ]
                         ];

    frm_table = frm_names[ [ pf_standard,  'spf'    ],
                           [ pf_standard,  'rpw'    ],
                           [ pf_d1b,       'd1b'    ],
                           [ pf_d1ba,      'd1ba'   ],
                           [ pf_d1b,       'd20'    ],
                           [ pf_xydata,    'xydata' ],
                           [ pf_d5000,     'd5000'  ],
                           [ pf_macinelrx, 'inelrx' ],
                           [ pf_singleraw, 'sglraw' ],
                           [ pf_njafrm,    'nja'    ],
                           [ pf_column,    'column' ]
                         ];

    in_min = ORD( 'a' ) - ORD( 'A' );


  var
    i, ii, iv, j, jj, ll, nop, np, nv: integer;
    ch:                                   char;
    sopt, sparm:                 string( 255 );
    rvl:            array[1..opt_nval] of real;
    opt:                            opt_identy;
    frmf:                          file_format;
    helpf:                                text;
    bhelp, bm:                         boolean;


    procedure INIT_GET_VALUES( var nvl: integer; ll: integer; bfrac: boolean := false );
    const
      opttsz = 10;

    var
      ij, jj, kp, num, den:      integer;
      snum:                 string( 14 );
      ch:                           char;

    begin { INIT_GET_VALUES }
      nvl := 0;
      ij  := 1;
      while (ij <= sparm.length) and (nvl < ll) do
      begin
        jj := INDEX( sparm, ':', nvl + 1 );
        kp := INDEX( sparm, ',', nvl + 1 );
        if kp > 0 then
          if (jj <= 0) or (jj > kp) then jj := kp;
        if jj <= 0 then jj := sparm.length + 1;
        nvl := nvl + 1;
        rvl[nvl] := 0.0;
        if jj > ij then
        begin
          if bfrac then
          begin
            snum := SUBSTR( sparm, ij, jj-ij+1 );
            kp := INDEX( snum, '/' );
            if kp > 1 then
            begin
              READV( snum, num:kp-1, ch, den:snum.length-kp );
              if den < 1 then den := 1;
              rvl[nvl] := num/den
            end
            else READV( sparm:ij, rvl[nvl]:jj-ij+1 )
          end
          else READV( sparm:ij, rvl[nvl]:jj-ij+1 )
        end;
        ij := jj + 1
      end
    end INIT_GET_VALUES;


  begin { GET_PARAMETERS }
    funcs  :=    [];
    bhelp  := false;
    { Set the default mode }
    for ij := 1 to 2*npat_max do
    begin  tbknd[ij] := true; tbtyp[ij] := 1; tbsiz[ij] := 1.0; tbcol[ij] := 1; tbpat[ij] := nil  end;
    inpl_frm.length := 0;
    outl_frm.length := 0;

    { *** Task Argument Loop *** }
    nop := 0;
    np  := 0;
    ii  := 1;
    while ii < argc do
    begin
      sparm := argv[ii]^;
     if sparm.length > 0 then
        if sparm[1] = '-' then                  { It is an Option Name or abreviation }
        begin                                   { Separes the Option Name and the Option Parameter List }
          nop := nop + 1;                       { Update the option count }
          jj := INDEX( sparm, '=' );
          if jj > 0 then
          begin
            sopt  := SUBSTR( sparm, 1, jj - 1 );
            if sopt.length > opttab_sm then sopt.length := opttab_sm;
            sparm := SUBSTR( sparm, jj + 1 )
          end
          else
          begin  sopt := sparm; sparm.length := 0  end;

          { Set option name in minor case }
          for ic := 1 to sopt.length do
            if (sopt[ic] >= 'A') and (sopt[ic] <= 'Z') then sopt[ic] := CHR( ORD( sopt[ic] ) + in_min );

          { Looking for this option name in the option table }
          opt := opt_not_opt;                   { Assume help until shown otherwise }
          for ic := 1 to opttab_sz do
            with opt_table[ic] do
              if opt_name = sopt then opt := opt_iden;

          case opt of
            opt_inpfrm,
            opt_outfrm:
              if sparm.length > 0 then
              begin { Specify input/output File Format }
                i := INDEX( sparm, ':' );
                if i > 0 then
                begin
                  if opt = opt_inpfrm then inpl_frm := SUBSTR( sparm, i + 1 )
                                      else outl_frm := SUBSTR( sparm, i + 1 );
                  sparm.length := i - 1
                end;
                SET_CASE( sparm, false );
(*
                for ic := 1 to sparm.length do
                  if (sparm[ic] >= 'A') and (sparm[ic] <= 'Z') then sparm[ic] := CHR( ORD( sparm[ic] ) + in_min );
*)
                frmf := pf_null;
                for ic := 1 to frmtab_sz do
                  with frm_table[ic] do
                  begin
                    j := 1;
                    repeat
                    exit if frm_name[j] <> sparm[j];
(*                  exit if frm_name[j] <> sparm[j]; *)
                      j := j + 1
                    until j > frm_name.length;
                    if j > frm_name.length then
                      if frmf = pf_null then
                        frmf := frm_iden                        { First matching in format string search - OK }
                      else
                      begin                                     { Second matching - Ambigous format => ERROR }
                        WRITELN( ' *** RPW-DATA Command error : Ambiguous file format "', sparm, '. ***' );
                        PASCAL_EXIT( 2 )
                      end
                  end;
                if frmf = pf_null then
                begin
                  WRITELN( ' *** RPW-DATA Command error : Unknown file format "', sparm, '. ***' );
                  PASCAL_EXIT( 2 )
                end;
                if opt = opt_inpfrm then inp_frm := frmf
                                    else out_frm := frmf
              end;

            opt_outpat:
              begin { Specify an output file }
                out_name := sparm;
                funcs := funcs + [fc_write]
              end;


            opt_label:
              begin { Label Plot request }
                funcs := funcs + [fc_plot];
                if (sparm[1] = 'l') or (sparm[1] = 'L') then funcs := funcs + [fc_label]
              end;

            opt_axisz:
              with drw_parm do
              begin
                INIT_GET_VALUES( nv, 3 );
                if nv > 0 then xsz_r := rvl[1];
                if nv > 1 then ysz_r := rvl[2];
                if nv > 2 then zsz_r := rvl[3]
              end;

            opt_npatt:
              begin { Specify an sequence number pattern selection }
                INIT_GET_VALUES( nv, opt_nval );
                if nv > npat_max - ndia then nv := 2*npat_max - ndia;
                for ic := 1 to nv do
                begin
                  ndia := ndia + 1;
                  tbnp[ndia] := ROUND( rvl[ic] );
                end;
                funcs := funcs + [fc_ndia,fc_plot]
              end;

            opt_tempsel:
              begin { Specify a temperature pattern selection }
                i := 1; ndiat := 0;
                while ndiat < npat_max do
                begin
                  ndiat := ndiat + 1;
                  READV( sparm:i, tbtem[ndiat] );
                exit if (sparm[i] <> ':') and (sparm[i] <> ',');
                  i := i + 1
                end;
                funcs := funcs + [fc_ndia,fc_plot]
              end;

            opt_shift:
              with drw_parm do
              begin { Specify a 2D plot pattern shift }
                INIT_GET_VALUES( nv, 3 );
                if nv >= 1 then pcs_r := ABS( rvl[1] );
                if nv >= 2 then hls_r := rvl[2];
                if nv >= 3 then vls_r := rvl[2]
              end;

            opt_pcolor:
              begin
                i := 1; j := 1;
                while (i <= sparm.length) and (i > 0) do
                begin
                  READV( sparm:i, j );
                  if (j < 1) or (j > 2*npat_max) then j := 1;
                  tbcol[j] := 1;
                exit if (i >= sparm.length) or (i <= 0);
                exit if sparm[i] <> '-';
                  i := i + 1;
                  READV( sparm:i, tbcol[j] );
                exit if (i >= sparm.length) or (i <= 0);
                  if (tbcol[j] < 1) or (tbcol[j] > 8) then tbcol[j] := 1;
                exit if (sparm[i] <> ':') and (sparm[i] <> ',');
                  i := i + 1
                end
              end;

            opt_pltlin,
            opt_pltmrk:
              begin { Select kind of line or kind of marker for selected pattern }
                bm := (opt = opt_pltmrk);
                i := 1; j := 1;
                loop
                  READV( sparm:i, j );
                exit if i <= 0;
                  if (j < 1) or (j > 2*npat_max) then j := 1;
                  tbknd[j] := bm;
                  tbtyp[j] := 1;
                exit if sparm[i] <> '-';
                  i := i + 1;
                exit if i > sparm.length;
                  READV( sparm:i, tbtyp[j] );
                exit if i <= 0;
                  if sparm[i] = '-' then
                  begin  i := i + 1; READV( sparm:i, tbsiz[j] )  end;
                exit if i <= 0;
                exit if (sparm[i] <> ':') and (sparm[i] <> ',');
                  i := i + 1;
                exit if i > sparm.length;
                end
              end;

            opt_taxis: { Swap to Celcius or Kelvin display }
              if sparm[1] = 'c' then funcs := funcs + [fc_celsius]      { Set Celsius mode }
                                else funcs := funcs - [fc_celsius];     { Set Kelvin mode - the default) }

            opt_dir:            funcs := funcs + [fc_dir];      { Set the directory request }

            opt_lambda:
              begin { Specify a wave length(2)  and optionaly a scan 2 theta step}
                INIT_GET_VALUES( nv, 4 );
                if nv > 0 then
                with hkl_parm, mai_parm do
                begin
                  gw1_r := rvl[1];
                  if nv > 1 then gw2_r := rvl[2];
                  if nv > 2 then th0_r := rvl[3];
                  if nv > 3 then ths_r := rvl[4]
                end
              end;

            opt_thlim:
              begin { Specify a display 2 theta range for 2D Plot }
                INIT_GET_VALUES( nv, 3 );
                if nv > 0 then
                with mai_parm do
                begin
                  tmi_r := rvl[1];
                  if nv > 1 then tma_r := rvl[2];
                end
              end;

            opt_deftemp:
              begin
                INIT_GET_VALUES( nv, 1 );
                dat_celsius := (rvl[1] >= 0.5)
              end;

            opt_verbose:        funcs := funcs + [fc_verbose];  { Set the verbose flag }

            opt_term:           mod_tty         := true;        { Set the terminal mode do not try to start the DRAW System) }

            opt_help:           bhelp           := true;        { Set the Help flag }

          otherwise
            WRITELN( ' *** RPW-DATA Command error : Unknown option "', sopt, '. ***' );
            bhelp := true;
            goto ET_HELP
          end;
        end
        else
        begin { Not an option }
          np := np + 1;
          case np of
            1:  inp_name := sparm;                              { Set the Job Name and default for other files }
            2:  begin
                  out_name := sparm;                            { Set the output file specification ... }
                  funcs := funcs + [fc_write]                   { ... and the file output flag }
                end;
          otherwise
            WRITELN( ' *** RPW-DATA Command warning : The file specification "', sparm, '"  will be ignored. ***' )
          end
        end;
      ii := ii + 1
    end { *** of Task Argument Loop *** };

    if mod_tty and (fc_plot in funcs) then
    begin
      WRITELN( ' *** RPW-DATA run in terminal mode. ***' );
      funcs := funcs - [fc_plot]
    end;


ET_HELP:
    { Check for complet Requirements }
    if bhelp then
    begin
      WRITELN( ' *** The RPW-DATA command model is :' );
      WRITELN( ' ':4, 'plot_pattern <option_list> <input_file> [<output_file>]' );
      WRITELN;
      WRITELN( ' Where : <option_list> is a list of                                                               short option name' );
      WRITELN( '    -setup=<expr>                  To force the use of a specific setup of filename "rpw_<expr>_setup.rpw_data"' );
      WRITELN( '                                   It is usefull to select an experiment suitable setup file.' );
      WRITELN( '    -terminal                      To force the terminal mode (without graphic). Usefull for a batch,      -tt' );
      WRITELN( '    -output=<output_file>          For an alternate specification of a pattern output file,                -o' );
      WRITELN( '    -inp_format=<iformat>          For a format specification of input file (default is d1b),              -if' );
      WRITELN( '    -out_format=<oformat>          For a format specification of output file (default is d1b),             -of' );
      WRITELN( '    -directory                     To get the directory list of the file (Pattern characteristics),         -d' );
      WRITELN( '    -t_unit_axis={C|K}             The temperature unit in Celcius (C) or Kelvin (K) - the default,        -tu' );
      WRITELN( '    -3dplot[=Light]                To generate a 3D (temperature dep. plot) with optional Light,           -3d' );
      WRITELN( '    -label                         To generate a 2D (temperature dep. plot) with opt. pattern label,       -lab');
      WRITELN( '    -axis_sizes=<x>[:<y>[:<z>]]    To specify prefered axis size (in cm) for x, y (and z with 3D plot),    -ax' );
      WRITELN( '    -pn_select=<pn1>[:<pn2> ...]   To select 2D plot pattern(s) by order number,                           -ns' );
      WRITELN( '    -pt_select=<pt1>[:<pt2> ...]   To select 2D plot pattern(s) by temperature value <ptx> selection,      -ts' );
      WRITELN( '    -pattern_shift=<shift_val>[:<label_shift>]                                                             -psh' );
      WRITELN( '                                   To set <shift_val> as the count rate of count range shift between' );
      WRITELN( '                                   two successive Pattern and the vertical <label_shift> in cm,' );
      WRITELN( '    -pattern_color:<cb>-<cl>[:<cb>-<cl>...]                                                                -pc' );
      WRITELN( '                                   To set the color <cl> at the specified pattern <cb>.' );
      WRITELN( '    -marker_kind:<cb>-<nt>[-<sz>][:<cb>-<nt>[-<sz>]...]                                                    -mk' );
      WRITELN( '                                   To set the kind of Marker <nt> for the pattern <cb> with size <sz>.' );
      WRITELN( '    -line_kind:<cb>-<nt>[-<sz>][:<cb>-<nt>[-<sz>]...]                                                      -lk' );
      WRITELN( '                                   To set the kind of Line <nt> for the pattern <cb> with size <sz>.' );
      WRITELN( '    -wave_length=<lambda>,[<torg>[:<tstep>]]                                                               -w' );
      WRITELN( '                                   To force a wave length of <lambda> in Angstroem and optional 2theta' );
      WRITELN( '                                   origine and step (in decimal degrees),' );
      WRITELN( '    -theta_range=<tmin>[:<tmax>]   To set a displayed 2theta range to be [<tmin>..<tmax>].                 -th' );
      WRITELN;
      WRITELN( ' The possible value for <iformat> and <oformat> are : inelxr, d1b, d20, xydata and column respectively for' );
      WRITELN( '        inelrx        X-ray data from Inel J. Marcus diffractometer (only for use with input as "-inp_format=inelrx"),' );
      WRITELN( '        D1B, D20      ILL neutron diffractometer format (D1B and D20 are same), It is the default for input and output,' );
      WRITELN( '        SPF           (or RPW) Our Standard format (close to XYDATA but with multi pattern management,' ); 
{     WRITELN( '        D1BA          ILL D1B alternative specific data format (with eff. factor numor info in the data)' ); }
      WRITELN( '        xydata        Standard XYDATA format that can be used only for output with "-out_format=xydata",' );
      WRITELN( '        column        Column format (Output is a file with number in column) to use with many plot and computing programs.' );
      WRITELN;
      WRITELN( ' The <pn1>, <pn2> ... values (integer numbers) are the ordering numbers of patterns in the input file.' );
      WRITELN( ' The <pt1>, <pt2> ... values (floating numbers) are the temperatures of patterns to select in the input file.' );
      WRITELN( ' The <shift_val>      value (floating number) is the shift (in count value) between to successive selected pattern to plot.' );
      WRITELN( ' The <cb>             values (integer numbers) are the sequence number of selected pattern to plot.' );
      WRITELN( ' The <cl>             values correspond to the color number as 0-white, 1-black, 2-red, 3-blue, 4-green, 5-yellow, 6-cyan, 7-magenta.' );
      WRITELN( ' The <nt>             values are corresponding to the numebr of marker or line kind (in agreement with the DRAW manual).' );
      WRITELN( ' The <sz>             values (floating numbers) are the thickness of lines or size of markers.' ); 
      WRITELN;
      if opt <> opt_not_opt then PASCAL_EXIT( 0 )
                            else PASCAL_EXIT( 2 )
    end

  end GET_PARAMETERS;




  procedure OUT_CATALOGUE( bsel: boolean );
  var
    p: pat_ptr;
    ct:   char;
    tm:   real;

  procedure OUT_INFO( p: pat_ptr; tc: boolean );
  var
    ct: char;

  begin
    with p^ do
    begin
      if tc then begin tm := t_sample - Ice_Temp; ct := 'C' end
            else begin tm := t_sample; ct := 'K' end;
      WRITELN( ' #', numor:-4, ' /  T = ', tm:8:2, ct, ', Sample = "', sample, '",' );
      WRITELN( ' ':10, 'Comment = "', comment, '",' );
      WRITELN( ' ':10, 'Mon = ', monitor:12:0, ', Time = ', cnttime:12:3, ', date/hour = "', dateh, '"' );
    end
  end OUT_INFO;

  begin { OUT_CATALOGUE }
    WRITELN;
    if bsel then
    begin
      WRITELN( ' ':5, 'List of Selected Diffraction Pattern from the file "', inp_name, '" :' );
      WRITELN;
      for ij := 1 to ndia do
        OUT_INFO( tbpat[ij], fc_celsius in funcs )
    end
    else
    begin
      WRITELN( ' ':5, 'List of Diffraction Pattern find in the file "', inp_name, '" :' );
      WRITELN;
      p := pat_first;
      while p <> nil do
      begin
        OUT_INFO( p, fc_celsius in funcs );
        p := p^.next
      end
    end;
    WRITELN
  end OUT_CATALOGUE;



[global 'RPW_Set_Ref']
procedure SET_REF( var p: $wild_pointer; var t: $wild_thing );
begin
  p := t"address
end SET_REF;



[global]
function  WRITE_SETUP_FILE( in_var file_spc: string; var envtb: pdsc_env ): ^string;
var
  pout: text;

begin
  OPEN( pout, file_spc, [write_file,error_file] );
  if iostatus = 0 then
  begin
    for ie := 0 to envtb.sz do
      if envtb[ie] <> nil then
        with envtb[ie]^ do
        begin
          WRITELN( pout, 'Environment ', ti );
          for it := 1 to ni do
            with tb[it] do
              case kn of
                prm_bool: WRITELN( pout, nm:5, ' = ', ORD( rb^ ):0 );
                prm_enm:  if el <> nil then WRITELN( pout, nm:5, ' = ', el^[ei^+1] )
                                       else WRITELN( pout, nm:5, ' = ', ei^ );
                prm_int:  WRITELN( pout, nm:5, ' = ', ri^ );
                prm_flt:  WRITELN( pout, nm:5, ' = ', rf^ );
                prm_str:  WRITELN( pout, nm:5, ' = ', rs^ );
              otherwise
              end;
          WRITELN( pout, 'End' );
          WRITELN( pout )
        end;
    CLOSE( pout );
    WRITE_SETUP_FILE := nil
  end
  else
  begin { Cannot open the setup file }
    WRITEV( str_msg, ' *** RPW Error: Cannot open the Setup file "', file_spc,
                     '" for output with err # ', iostatus:0, '. ***' );
    WRITE_SETUP_FILE := str_msg"address
  end
end WRITE_SETUP_FILE;



[global]
function  READ_SETUP_FILE( in_var file_spc: string; var envtb: pdsc_env ): ^string;
var
  bcont:       boolean;
  line:       linetype;
  pinp:           text;


  function  READ_SETUP_1ENV( var env: pdsc_tab ): boolean;
  var
    bcont:             boolean;
    ch:                   char;
    it, nn:            integer;
    senm, smod:      labeltype;

  begin
    bcont := false;
    with env do
    begin
(*
WRITELN( ' Read setup Env "', ti, '"' );
*)
      while not EOF( pinp ) do                          { Loop on all Setup Lines }
      begin
        READ( pinp, line:0:true );                      { Read one Setup Line identifier }
      exit if EOF( pinp );                              { Stop on End Of File }
        line := SUBSTR( line, 1, 5 );                   { Extract the five characters Identifier }
        bcont := (line = 'End');                        { Check for the end of Setup Environment }
      exit if bcont;
        it := 1;                                        { Looks for the specified identifier }
        while (it <= ni) and ((line <> SUBSTR( tb[it].nm, 1, 5 ) ) or (tb[it].kn <= prm_frm)) do  it := it + 1;
        if it <= ni then                                { When the identifier is found, Try to get its value }
        with tb[it] do
        begin
          ch := ' ';                                    { Look for the '=' separator }
          while not EOLN( pinp ) and (ch <> '=') do READ( pinp, ch );
          while not EOLN( pinp ) and (pinp^ = ' ') do GET( pinp );
          case kn of                                    { Dispatch with the identifier type }
            prm_bool: begin  READLN( pinp, it ); rb^ := (it > 0)  end;
            prm_enm:  begin (* WRITELN( ' *** Get ENM "', tb[it].nm, '"' ); *)
                        if el = nil then READLN( pinp, ei^ )
                        else
                        begin
                          READLN( pinp, senm:0:true ); (*  WRITELN( ' Value = "', senm, '"' ); *)
                          nn := 0;
                          if senm.length > 0 then
                          begin
                            SET_CASE( senm, false );
                            if (senm[1] >= 'a') and (senm[1] <= 'z') then
                            begin
                              repeat
                                smod := el^[nn+1];
                                SET_CASE( smod, false );
                                smod.length := senm.length;
                              exit if senm = smod;
                                nn := nn + 1
                              until nn >= el^.dim;
                              if nn >= el^.dim then nn := 0
                            end
                            else READV( senm, ei^ )
                          end;
                          ei^ := nn
                        end;
                      end;
            prm_int:  READLN( pinp, ri^ );
            prm_flt:  READLN( pinp, rf^ );
            prm_str:  if EOLN( pinp ) then rs^.length := 0
                                      else READLN( pinp, rs^ );
          otherwise
            READLN( pinp )
          end
        end
        else READLN( pinp )
      end
    end;
    READ_SETUP_1ENV := bcont
  end READ_SETUP_1ENV;


begin { READ_SETUP_FILE }
(*
  WRITELN( ' Read Setup file "', file_spc, '"' );
*)
  OPEN( pinp, file_spc, [read_file, error_file] );
  if iostatus = 0 then
  begin
    bcont := true;
    while not EOF( pinp ) and bcont do
    begin
      READ( pinp, line:0:true );
      if line = 'Environment' then
      begin
        READLN( pinp, line:0:true );
        line := SUBSTR( line, 1, 8 );
        for ie := 0 to envtb.sz do
          if envtb[ie] <> nil then
            if line = SUBSTR( envtb[ie]^.ti, 1, line.length ) then bcont := READ_SETUP_1ENV( envtb[ie]^ );
      end
    end;
    CLOSE( pinp );
    READ_SETUP_FILE := nil
  end
  else
  begin { Cannot open the setup file }
    WRITEV( str_msg, ' *** RPW Error: Cannot open the Setup file "', file_spc,
                     '" for input with err # ', iostatus:0, '. ***' );
    READ_SETUP_FILE := str_msg"address
  end
end READ_SETUP_FILE;



procedure SETUP_PATH_SEARCH;
var
  upath, dir, nam:      string;
  ib, ie, n:           integer;
  fnd:                 boolean;

begin
  if sys_unix <> 0 then upath := unix_setup_path
                   else upath := win_setup_path;
  ib  :=     1;
  n   :=     1;
  fnd := false;
  if setupf.length > 0 then nam := setupf
                       else nam := Const_def_setup;
  while not fnd and (ib <> 0) do
  begin
    ie := INDEX( upath, ';', n ); n := n + 1;
    if ie = 0 then
    begin  dir := SUBSTR( upath, ib ); ib := 0  end
    else
    begin  dir := SUBSTR( upath, ib, ie - ib ); ib := ie + 1  end;
    if dir[dir.length] <> '/' then begin  dir.length := dir.length + 1; dir[dir.length] := '/'  end;
    default_setup := dir||nam;
    fnd := FILE_ACCESS_CHECK( default_setup, 4 {Read access} )
  end;
  if not fnd then
  begin
    if spc_stf then WRITEV( str_msg, ' *** RPW-DATA cannot open the setup file "', default_setup, '"' );
    default_setup := Const_def_setup
  end;
  setupf := default_setup
end SETUP_PATH_SEARCH;



begin { Main - RPW_DATA }
  MAIN$SETUP_INIT;                              { Initialize all setup records }
  EXTRACT_SETUP_OPTION;                         { Handle the setup option in first }
  SETUP_PATH_SEARCH;                            { Look for a default setup file specification }
  READ_SETUP_FILE( setupf, setup_tab );         { Initial Setup Loading }

  { Initialize the program and get the specified files and options }
  INIT_LOAD;
  GET_PARAMETERS;

  { Get All Diffraction Patterns }
  if inp_name.length > 0 then GET_PATTERNS;

  { Do a 2D Plot when required }
(*
  if not mod_tty and ([fc_ndia,fc_plot]*funcs = [fc_ndia,fc_plot]) and (ndia > 0) then
*)
  if not mod_tty then
  begin
    MAKE_2D( ndia );
    if not drw_ok then
    begin
      WRITELN( ' *** RPW-DATA cannot start th DRAW System: The graphic function cannot be used in this environment. ***' );
      mod_tty := true
    end
  end;

  if mod_tty then
  begin
    if str_msg.length > 0 then
    begin
      WRITELN( str_msg );
      WRITELN( ' *** RPW-DATA Stop on Error.' );
      PASCAL_EXIT( 2 )
    end
  end;

  if fc_dir in funcs then OUT_CATALOGUE( ndia <> 0  );

  { Create the Output when required }
  if fc_write in funcs then WRITE_NORMALIZED_FILE;

  WRITELN( ' Normal End.' )

end RPW_DATA.

